/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

define([
    'jquery',
    'mage/translate',
    'collapsible'
], function ($, $t) {
    'use strict';

    /**
     * Bind settings dropdown toggle.
     */
    function bindCollapsible(self)
    {
        var collapsibleSelector = '.wtai-custom-action-select-wrap';

        $(collapsibleSelector).collapsible({
            collapsible: true,
            openedState: '_active',
            active: false
        });

        $(collapsibleSelector).on('beforeOpen', function () {
            $(collapsibleSelector).not(this).collapsible('deactivate');
            if ($(this).hasClass('wtai-audience')) {
                self.getAudience();
            }
        });

        $(document).on('click', function (e) {
            if (!$(collapsibleSelector).is(e.target) &&
                !$(collapsibleSelector).has(e.target).length &&
                $(collapsibleSelector).data('collapsible')
            ) {
                $(collapsibleSelector).collapsible('deactivate');
            }
        });
    }

    /**
     * Check if combination is not allowed
     *
     * @param {string} id
     * @param {string} type
     *
     * @returns {boolean}
     */
    function isCombinationDisallowed(id, type, self)
    {
        var result,
            isStyleChecked = false,
            isToneChecked = false,
            isAudienceChecked = false,
            combinationHasAudience = false;

        try {
            if (type !== 'Audience' && !self.editToneStyleAllowed) {
                return true;
            }

            /* Check if combination is available */
            result = self
                .disallowedCombinations()
                .find(function (combination) {
                    return combination.combination.some(function (obj) {
                        return obj.type === type && obj.id === id;
                    });
                });

            if (!(self.disallowedCombinations() && result)) {
                return false;
            }

            switch (type) {
                case 'Tone':
                    isStyleChecked = result.combination.some(
                        function (obj) {
                            return (
                                obj.type === 'Style' &&
                                self.stylesSelected() === obj.id
                            );
                        }
                    );
                    combinationHasAudience =
                        result.combination.some(function (obj) {
                            return obj.type === 'Audience';
                        });

                    if (combinationHasAudience) {
                        isAudienceChecked =
                            self
                                .audienceSelected()
                                .indexOf(combinationHasAudience.id) !== -1;
                        
                        return isStyleChecked && isAudienceChecked;
                    }
                    return isStyleChecked;

                case 'Style':
                    isToneChecked = result.combination.some(
                        function (obj) {
                            return (
                                obj.type === 'Tone' &&
                                self.tonesSelected().indexOf(obj.id) !== -1
                            );
                        }
                    );
                    combinationHasAudience =
                        result.combination.some(function (obj) {
                            return obj.type === 'Audience';
                        });

                    if (combinationHasAudience) {
                        isAudienceChecked =
                            self
                                .audienceSelected()
                                .indexOf(combinationHasAudience.id) !== -1;

                        return isToneChecked && isAudienceChecked;
                    }
                    return isToneChecked;

                case 'Audience':
                    isToneChecked = result.combination.some(
                        function (obj) {
                            return (
                                obj.type === 'Tone' &&
                                self.tonesSelected().indexOf(obj.id) !== -1
                            );
                        }
                    );
                    isStyleChecked = result.combination.some(
                        function (obj) {
                            return (
                                obj.type === 'Style' &&
                                self.stylesSelected() === obj.id
                            );
                        }
                    );

                    return isStyleChecked && isToneChecked;

                default:
                    return false;
            }
        } catch (error) {
            return false;
        }
    }

    /**
     * Process tooltip content
     *
     * @param {String} selector
     * @param {String} stringValue
     *
     * @returns {String}
     */
    function processToolTip(stringValue, type)
    {
        var textBase = '';
        switch (type) {
            case 'Tone':
                textBase = $t("Unavailable when the '%1' tone is selected.");
                break;
            case 'Style':
                textBase = $t("Unavailable when the '%1' style is selected.");
                break;
            default:
                textBase = $t("Unavailable when the '%1' audience is selected.");
                break;
        }
        
        return $t(textBase).replace('%1', stringValue);
    }

    /**
     * Get tooltip text based on disallowed combination
     *
     * @param {string} id
     * @param {string} type
     *
     * @returns {string}
     */
    function getTooltipCombinationDisallowedText(id, type, self)
    {
        var result,
            isStyleChecked = false,
            isToneChecked = false,
            textBase = '';

        try {
            /* Check if combination is available */
            result = self
                .disallowedCombinations()
                .find(function (combination) {
                    return combination.combination.some(function (obj) {
                        return obj.type === type && obj.id === id;
                    });
                });

            if (!result) {
                return '';
            }
            
            switch (type) {
                case 'Tone':
                    isStyleChecked = result.combination.some(
                        function (obj) {
                            if (obj.type === 'Style' && self.stylesSelected() === obj.id) {
                                textBase = processToolTip(obj.id, 'Style');
                                return true;
                            }
                            return false;
                        }
                    );
                    return textBase;

                case 'Style':
                    isToneChecked = result.combination.some(
                        function (obj) {
                            if ( obj.type === 'Tone' &&
                                self.tonesSelected().indexOf(obj.id) !== -1) {
                                textBase = processToolTip(obj.id, 'Tone');
                                return true;
                            }
                            return false;
                        }
                    );
                    return textBase;

                default:
                    return '';
            }
        } catch (error) {
            return '';
        }
    }

    return {
        bindCollapsible: bindCollapsible,
        isCombinationDisallowed: isCombinationDisallowed,
        getTooltipCombinationDisallowedText: getTooltipCombinationDisallowedText
    }
});