<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\Data\NewRecordsInterface;
use WriteTextAI\WriteTextAI\Api\Data\NewRecordsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\NewRecordsSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\NewRecordsRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ResourceModel\NewRecords as ResourceNewRecords;
use WriteTextAI\WriteTextAI\Model\ResourceModel\NewRecords\CollectionFactory as NewRecordsCollectionFactory;

class NewRecordsRepository implements NewRecordsRepositoryInterface
{

    /**
     * @var NewRecordsInterfaceFactory
     */
    protected $newRecordsFactory;

    /**
     * @var NewRecordsCollectionFactory
     */
    protected $newRecordsCollectionFactory;

    /**
     * @var NewRecords
     */
    protected $searchResultsFactory;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var ResourceNewRecords
     */
    protected $resource;

    /**
     * @param ResourceNewRecords $resource
     * @param NewRecordsInterfaceFactory $newRecordsFactory
     * @param NewRecordsCollectionFactory $newRecordsCollectionFactory
     * @param NewRecordsSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceNewRecords $resource,
        NewRecordsInterfaceFactory $newRecordsFactory,
        NewRecordsCollectionFactory $newRecordsCollectionFactory,
        NewRecordsSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->newRecordsFactory = $newRecordsFactory;
        $this->newRecordsCollectionFactory = $newRecordsCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(NewRecordsInterface $newRecords)
    {
        try {
            $this->resource->save($newRecords);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the newRecords: %1',
                $exception->getMessage()
            ));
        }
        return $newRecords;
    }

    /**
     * @inheritDoc
     */
    public function get($id)
    {
        $newRecords = $this->newRecordsFactory->create();
        $this->resource->load($newRecords, $id);
        if (!$newRecords->getId()) {
            throw new NoSuchEntityException(__('NewRecords with id "%1" does not exist.', $id));
        }
        return $newRecords;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->newRecordsCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(NewRecordsInterface $newRecords)
    {
        try {
            $newRecordsModel = $this->newRecordsFactory->create();
            $this->resource->load($newRecordsModel, $newRecords->getId());
            $this->resource->delete($newRecordsModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the NewRecords: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($id)
    {
        return $this->delete($this->get($id));
    }
}
