<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Model;

use WriteTextAI\WriteTextAI\Model\FullAutomationRequestsFactory;
use WriteTextAI\WriteTextAI\Api\FullAutomationRequestsRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ApiManager;
use WriteTextAI\WriteTextAI\Model\Api\Keywords;

class FullAutomationRequestsManager
{
    /**
     * @var FullAutomationRequestsFactory
     */
    protected $FullAutomationRequestsFactory;

    /**
     * @var FullAutomationRequestsRepositoryInterface
     */
    protected $FullAutomationRequestsRepository;

    /**
     * @var ApiManager
     */
    protected $apiManager;

    /**
     * @var Keywords
     */
    protected $keywords;

    /**
     * Constructor
     *
     * @param FullAutomationRequestsFactory $FullAutomationRequestsFactory
     * @param FullAutomationRequestsRepositoryInterface $FullAutomationRequestsRepository
     * @param ApiManager $apiManager
     * @param Keywords $keywords
     */
    public function __construct(
        FullAutomationRequestsFactory $FullAutomationRequestsFactory,
        FullAutomationRequestsRepositoryInterface $FullAutomationRequestsRepository,
        ApiManager $apiManager,
        Keywords $keywords
    ) {
        $this->FullAutomationRequestsFactory = $FullAutomationRequestsFactory;
        $this->FullAutomationRequestsRepository = $FullAutomationRequestsRepository;
        $this->apiManager = $apiManager;
        $this->keywords = $keywords;
    }

    /**
     * Set full automation request
     *
     * @param string $requestId
     * @param string $user
     * @param string $entityType
     * @param string $storeId
     * @param string $recordId = null
     * @return void
     */
    public function setFullAutomationRequest($requestId, $user, $entityType, $storeId, $recordId = null)
    {
        $FullAutomationRequests = $this->FullAutomationRequestsFactory->create();

        $existing = $FullAutomationRequests->getCollection()
            ->addFieldToFilter('user', $user)
            ->getFirstItem();

        if ($existing->getId()) {
            // Update the existing record
            $existing->setRequestId($requestId);
            $existing->setEntityType($entityType);
            $existing->setStoreId($storeId);
            $existing->setRecordId($recordId);
            $FullAutomationRequests = $existing;
        } else {
            // Record not found, create a new one
            $FullAutomationRequests->setData([
                "user" => $user,
                "request_id" => $requestId,
                "entity_type" => $entityType,
                "store_id" => $storeId,
                "record_id" => $recordId
            ]);
        }

        $this->FullAutomationRequestsRepository->save($FullAutomationRequests);
    }

    /**
     * Delete full automation request
     *
     * @param string $requestId
     * @return void
     */
    public function delete($requestId)
    {
        $FullAutomationRequests = $this->FullAutomationRequestsFactory->create();

        $existing = $FullAutomationRequests->getCollection()
            ->addFieldToFilter('request_id', $requestId)
            ->getFirstItem();

        if ($existing->getId()) {
            $this->FullAutomationRequestsRepository->delete($existing);
        }
    }

    /**
     * Delete full automation request by user
     *
     * @param string $user
     * @return void
     */
    public function deleteByUser($user)
    {
        $FullAutomationRequests = $this->FullAutomationRequestsFactory->create();

        $existing = $FullAutomationRequests->getCollection()
            ->addFieldToFilter('user', $user)
            ->getFirstItem();

        if ($existing->getId()) {
            $this->FullAutomationRequestsRepository->delete($existing);
        }
    }

    /**
     * Delete all full automation requests
     *
     * @return void
     */
    public function deleteAll()
    {
        $FullAutomationRequests = $this->FullAutomationRequestsFactory->create();
        $FullAutomationRequests->getCollection()->walk('delete');
    }

    /**
     * Get full automation requests
     *
     * @return bool
     */
    public function getFullAutomationRequests()
    {
        $FullAutomationRequests = $this->FullAutomationRequestsFactory->create();
        $FullAutomationRequests = $FullAutomationRequests->getCollection();

        $requests = [];
        foreach ($FullAutomationRequests as $FullAutomationRequest) {
            try {
                $apiResponse = $this->apiManager->getBulkRequestById($FullAutomationRequest->getRequestId());
                if (!empty($apiResponse)) {
                    $failed = isset($apiResponse['failed']) ? $apiResponse['failed'] : 0;
                    $completed = $apiResponse['completed'] !== $failed
                        ? $apiResponse['completed'] - $failed
                        : $apiResponse['completed'];
                    
                    $status = [
                        'id' => $FullAutomationRequest->getRequestId(),
                        'message' => $this->getMessage(
                            $apiResponse['status'],
                            $apiResponse['failed'] ?? 0,
                            $apiResponse['total'] ?? 0
                        ),
                        'progress' => sprintf(
                            __('%s / %s processed and %s completed'),
                            $completed,
                            $apiResponse['total'],
                            $completed
                        ),
                        'percent' => floor($apiResponse['completed'] / $apiResponse['total'] * 100),
                        'completed' => $completed,
                        'status' => $apiResponse['status'],
                        'queuedIds' => $apiResponse['queuedIds'],
                        'completedIds' => $apiResponse['completedIds'],
                        'user' => $apiResponse['userName'],
                        'subType' => $apiResponse['subType'] ?? 2,
                        'failed' => $apiResponse['failed'] ?? 0,
                        'failedIds' => $apiResponse['failedIds'] ?? [],
                        'total' => $apiResponse['total'] ?? 0,
                        'runningIds' => $apiResponse['runningIds'] ?? []
                    ];
                }
            } catch (\WriteTextAI\WriteTextAI\Exception\ApiException $e) {
                if ($e->getCode() !== 404) {
                    throw $e;
                }
                $getOptimizationResponse = $this->keywords->getOptimizationList(
                    $FullAutomationRequest->getStoreId(),
                    ['Failed'],
                    null,
                    $FullAutomationRequest->getEntityType(),
                    [$FullAutomationRequest->getRecordId()]
                );
                $apiResponse = [
                    'status' => 'Completed',
                    'completed' => 1,
                    'total' => 1,
                    'queuedIds' => [],
                    'completedIds' => [],
                    'userName' => $FullAutomationRequest->getUser(),
                    'estimatedEndTime' => null
                ];
                if (isset($getOptimizationResponse['result'])) {
                    foreach ($getOptimizationResponse['result'] as $key => $value) {
                        if ($value['status'] === 'Failed') {
                            $apiResponse = [
                                'id' => $FullAutomationRequest->getRequestId(),
                                'message' => __('Full automation failed, you can restart to try again.'),
                                'status' => 'Completed',
                                'progress' => sprintf(
                                    __('%s / %s processed and %s completed'),
                                    1,
                                    1,
                                    1
                                ),
                                'percent' => $value['completedSteps'] / $value['totalSteps'] * 100,
                                'completed' => 1,
                                'total' => 1,
                                'queuedIds' => [],
                                'completedIds' => [],
                                'user' => $FullAutomationRequest->getUser(),
                                'subType' => 1,
                                'failed' => 1,
                                'failedIds' => [$value['recordId']],
                                'userName' => $FullAutomationRequest->getUser(),
                                'estimatedEndTime' => null,
                                'runningIds' => []
                            ];
                        }
                    }
                }
            }

            $remaining = $this->calculateRemainingTime($apiResponse['estimatedEndTime']);
            $message = $this->getMessage($apiResponse, $remaining);
            $failed = isset($apiResponse['failed']) ? $apiResponse['failed'] : 0;
            $status = $apiResponse['status'];

            if ($status === 'Failed') {
                $status = 'Completed';
            }
            
            if ($apiResponse['status'] === 'Failed'
                && $apiResponse['completed'] === 1 && $apiResponse['total'] === 1) {
                $failed = 1;
            }

            $completed = $apiResponse['completed'] !== $failed
                ? $apiResponse['completed'] - $failed
                : $apiResponse['completed'];

            $requests[] = [
                'id' => $FullAutomationRequest->getRequestId(),
                'message' => $message,
                'completed' => $completed,
                'progress' => sprintf(
                    __('%s / %s processed and %s completed'),
                    $completed,
                    $apiResponse['total'],
                    $completed
                ),
                'percent' => floor($apiResponse['completed'] / $apiResponse['total'] * 100),
                'status' => $status,
                'queuedIds' => $apiResponse['queuedIds'],
                'completedIds' => $apiResponse['completedIds'],
                'completed' => $apiResponse['completed'],
                'user' => $apiResponse['userName'],
                'subType' => $apiResponse['subType'] ?? 0,
                'failed' => $failed,
                'failedIds' => $apiResponse['failedIds'] ?? [],
                'total' => $apiResponse['total'] ?? 0,
                'runningIds' => $apiResponse['runningIds'] ?? []
            ];
        }

        return $requests;
    }

    /**
     * Calculate remaining time
     *
     * @param string $estimatedEndTime
     * @return array
     */
    protected function calculateRemainingTime($estimatedEndTime)
    {
        $remaining = 0;
        $isMinutes = false;
        $result = [];
        $currentDatetime;
        $interval;

        if ($estimatedEndTime !== null) {
            $estimatedEndTime = new \DateTime($estimatedEndTime);
            $currentDatetime = new \DateTime();
            $interval = $estimatedEndTime->getTimestamp() - $currentDatetime->getTimestamp();
            // Convert seconds to minutes if it's more than or equal to 60 seconds
            $remaining = ceil($interval / 60);

            if ($remaining >= 60) {
                // Convert minutes to hours if it's more than or equal to 60 minutes
                $remaining = ceil($remaining / 60);
                $isMinutes = true;
            }
        }

        $result['remaining'] = $remaining;
        $result['is_minutes'] = $isMinutes;

        return $result;
    }

    /**
     * Get message
     *
     * @param array $apiResponse
     * @param array $remainingObj
     *
     * @return {String}
     */
    protected function getMessage($apiResponse, $remainingObj)
    {
        $message = '';
        $text = '';
        $failed = $apiResponse['failed'] ?? 0;
        $total = $apiResponse['total'] ?? 0;
        $status = $apiResponse['status'] ?? '';
        $remaining = $remainingObj['remaining'] ?? 0;
        $isMinutes = $remainingObj['is_minutes'] ?? false;
        $runningIds = $apiResponse['runningIds'] ?? [];

        if ($failed === $total && $failed > 0) {
            return sprintf(__('Full automation failed, you can restart to try again.'));
        }
        switch ($status) {
            case 'Pending':
                $message = __('Preparing for full automation...');
                break;
            case 'Running':
                if ($remaining === 0) {
                    $message = __('Running full automation...');
                    break;
                }
                if ($isMinutes) {
                    $text = __('Running full automation... (Estimated time remaining %s minute/s)');
                } else {
                    $text = __('Running full automation... (Estimated time remaining %s second/s)');
                }
                $message = sprintf($text, $remaining);
                break;
            case 'Completed':
                $message = __('WriteText.ai is done running full automation for your selected products.');
                break;
            case 'Cancelling':
                $message = __('Some products have already been queued up for processing.'
                    . ' Cancelling unqueued items only…');
                break;
            case 'Cancelled':
                $message = __('This process has been cancelled.');
                break;
            case 'TimedOut':
                $message = __('WriteText.ai is currently experiencing a high volume of requests and has timed out.');
                break;
            case 'Failed':
                $message = __('Error encountered—some products were not processed');
                break;
        }

        return $message;
    }
}
