<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\BulkGenerateRequestsRepositoryInterface;
use WriteTextAI\WriteTextAI\Api\Data\BulkGenerateRequestsInterface;
use WriteTextAI\WriteTextAI\Api\Data\BulkGenerateRequestsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\BulkGenerateRequestsSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Model\ResourceModel\BulkGenerateRequests as ResourceBulkGenerateRequests;
use WriteTextAI\WriteTextAI\Model\ResourceModel\BulkGenerateRequests\CollectionFactory
    as BulkGenerateRequestsCollectionFactory;

class BulkGenerateRequestsRepository implements BulkGenerateRequestsRepositoryInterface
{

    /**
     * @var ResourceBulkGenerateRequests
     */
    protected $resource;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var BulkGenerateRequests
     */
    protected $searchResultsFactory;

    /**
     * @var BulkGenerateRequestsCollectionFactory
     */
    protected $bulkGenerateRequestsCollectionFactory;

    /**
     * @var BulkGenerateRequestsInterfaceFactory
     */
    protected $bulkGenerateRequestsFactory;

    /**
     * @param ResourceBulkGenerateRequests $resource
     * @param BulkGenerateRequestsInterfaceFactory $bulkGenerateRequestsFactory
     * @param BulkGenerateRequestsCollectionFactory $bulkGenerateRequestsCollectionFactory
     * @param BulkGenerateRequestsSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceBulkGenerateRequests $resource,
        BulkGenerateRequestsInterfaceFactory $bulkGenerateRequestsFactory,
        BulkGenerateRequestsCollectionFactory $bulkGenerateRequestsCollectionFactory,
        BulkGenerateRequestsSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->bulkGenerateRequestsFactory = $bulkGenerateRequestsFactory;
        $this->bulkGenerateRequestsCollectionFactory = $bulkGenerateRequestsCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(
        BulkGenerateRequestsInterface $bulkGenerateRequests
    ) {
        try {
            $this->resource->save($bulkGenerateRequests);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the bulkGenerateRequests: %1',
                $exception->getMessage()
            ));
        }
        return $bulkGenerateRequests;
    }

    /**
     * @inheritDoc
     */
    public function get($bulkGenerateRequestsId)
    {
        $bulkGenerateRequests = $this->bulkGenerateRequestsFactory->create();
        $this->resource->load($bulkGenerateRequests, $bulkGenerateRequestsId);
        if (!$bulkGenerateRequests->getId()) {
            throw new NoSuchEntityException(__(
                'bulk_generate_requests with id "%1" does not exist.',
                $bulkGenerateRequestsId
            ));
        }
        return $bulkGenerateRequests;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->bulkGenerateRequestsCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(
        BulkGenerateRequestsInterface $bulkGenerateRequests
    ) {
        try {
            $bulkGenerateRequestsModel = $this->bulkGenerateRequestsFactory->create();
            $this->resource->load($bulkGenerateRequestsModel, $bulkGenerateRequests->getBulkGenerateRequestsId());
            $this->resource->delete($bulkGenerateRequestsModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the bulk_generate_requests: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($bulkGenerateRequestsId)
    {
        return $this->delete($this->get($bulkGenerateRequestsId));
    }
}
