<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */
declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\AiCategoryRepositoryInterface;
use WriteTextAI\WriteTextAI\Api\Data\AiCategoryInterface;
use WriteTextAI\WriteTextAI\Api\Data\AiCategoryInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\AiCategorySearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Model\ResourceModel\AiCategory as ResourceAiCategory;
use WriteTextAI\WriteTextAI\Model\ResourceModel\AiCategory\CollectionFactory as AiCategoryCollectionFactory;

class AiCategoryRepository implements AiCategoryRepositoryInterface
{
    /**
     * @var AiCategoryCollectionFactory
     */
    protected $aiCategoryCollectionFactory;

    /**
     * @var AiCategory
     */
    protected $searchResultsFactory;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var AiCategoryInterfaceFactory
     */
    protected $aiCategoryFactory;

    /**
     * @var ResourceAiCategory
     */
    protected $resource;

    /**
     * @param ResourceAiCategory $resource
     * @param AiCategoryInterfaceFactory $aiCategoryFactory
     * @param AiCategoryCollectionFactory $aiCategoryCollectionFactory
     * @param AiCategorySearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceAiCategory $resource,
        AiCategoryInterfaceFactory $aiCategoryFactory,
        AiCategoryCollectionFactory $aiCategoryCollectionFactory,
        AiCategorySearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->aiCategoryFactory = $aiCategoryFactory;
        $this->aiCategoryCollectionFactory = $aiCategoryCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(AiCategoryInterface $aiCategory)
    {
        try {
            $this->resource->save($aiCategory);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the aiCategory: %1',
                $exception->getMessage()
            ));
        }
        return $aiCategory;
    }

    /**
     * @inheritDoc
     */
    public function get($aiCategoryId)
    {
        $aiCategory = $this->aiCategoryFactory->create();
        $this->resource->load($aiCategory, $aiCategoryId);
        if (!$aiCategory->getId()) {
            throw new NoSuchEntityException(__('AiCategory with id "%1" does not exist.', $aiCategoryId));
        }
        return $aiCategory;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->aiCategoryCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(AiCategoryInterface $aiCategory)
    {
        try {
            $aiCategoryModel = $this->aiCategoryFactory->create();
            $this->resource->load($aiCategoryModel, $aiCategory->getAicategoryId());
            $this->resource->delete($aiCategoryModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the AiCategory: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($aiCategoryId)
    {
        return $this->delete($this->get($aiCategoryId));
    }
}
