<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Helper;

use Magento\Framework\App\Cache\Manager;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\App\Helper\Context;
use Magento\Store\Model\ScopeInterface;
use WriteTextAI\WriteTextAI\Model\Config\Source\AutomaticTextOptimization;
use WriteTextAI\WriteTextAI\Model\Config\Source\ProductAttributes\Text as TextTypeAttributes;
use WriteTextAI\WriteTextAI\Model\Config\Source\CategoryAttributes\Text as CategoryTextTypeAttributes;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Helper\Output as CatalogOutput;

class Settings extends \Magento\Framework\App\Helper\AbstractHelper
{
    public const SECTION_NAME = 'writetextai_settings';

    public const API_SETTINGS_CONFIG_PATH = 'writetextai_settings/api/';

    public const GENERAL_SETTINGS_CONFIG_PATH = 'writetextai_settings/general';

    public const ATTRIBUTES_SETTINGS_CONFIG_PATH = 'writetextai_settings/attributes/';

    public const DESCRIPTION_SETTINGS_CONFIG_PATH = 'writetextai_settings/description/';

    public const EXCERPT_SETTINGS_CONFIG_PATH = 'writetextai_settings/excerpt/';

    public const CATEGORY_DESCRIPTION_SETTINGS_CONFIG_PATH = 'writetextai_settings/category_description/';

    public const MAPPING_SETTINGS_CONFIG_PATH = 'writetextai_settings/mapping';

    public const CRON_SETTINGS_CONFIG_PATH = 'writetextai_settings/cron/';

    public const SETUP_SETTINGS_CONFIG_PATH = 'writetextai_settings/setup/';

    public const MAX_LENGTH_SETTINGS_CONFIG_PATH = 'writetextai_settings/max_length/';

    public const TRANSLATION_SETTINGS_CONFIG_PATH = 'writetextai_settings/translation/';

    public const KEYWORD_OPTIMIZATION_SETTINGS_CONFIG_PATH = 'writetextai_settings/keyword_optimization';

    public const DEV_SETTINGS_CONFIG_PATH = 'writetextai_settings/dev/';

    public const CANNIBALIZATION_SETTINGS_CONFIG_PATH = 'writetextai_settings/cannibalization/';

    public const AUTOTRANSFER_SETTINGS_CONFIG_PATH = 'writetextai_settings/autotransfer/';

    /**
     * @var Context
     */
    protected $context;

    /**
     * @var WriterInterface
     */
    protected $configWriter;

    /**
     * @var Manager
     */
    protected $cacheManager;

    /**
     * @var AutomaticTextOptimization
     */
    protected $automaticTextOptimization;

    /**
     * @var TextTypeAttributes
     */
    protected $textTypeAttributes;

    /**
     * @var CategoryTextTypeAttributes
     */
    protected $categoryTextTypeAttributes;

    /**
     * @var ProductCollectionFactory
     */
    protected $productCollectionFactory;

    /**
     * @var CatalogOutput
     */
    protected $catalogOutput;

    /**
     * @var array
     */
    protected $mappingSettings = [];

    /**
     * @param Context $context
     * @param WriterInterface $configWriter
     * @param Manager $cacheManager
     * @param AutomaticTextOptimization $automaticTextOptimization
     * @param TextTypeAttributes $textTypeAttributes
     * @param CategoryTextTypeAttributes $categoryTextTypeAttributes
     * @param ProductCollectionFactory $productCollectionFactory
     * @param CatalogOutput $catalogOutput
     */
    public function __construct(
        Context $context,
        WriterInterface $configWriter,
        Manager $cacheManager,
        AutomaticTextOptimization $automaticTextOptimization,
        TextTypeAttributes $textTypeAttributes,
        CategoryTextTypeAttributes $categoryTextTypeAttributes,
        ProductCollectionFactory $productCollectionFactory,
        CatalogOutput $catalogOutput
    ) {
        $this->context = $context;
        $this->configWriter = $configWriter;
        $this->cacheManager = $cacheManager;
        $this->automaticTextOptimization = $automaticTextOptimization;
        $this->textTypeAttributes = $textTypeAttributes;
        $this->categoryTextTypeAttributes = $categoryTextTypeAttributes;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->catalogOutput = $catalogOutput;
        parent::__construct($context);
    }
    
    /**
     * Get config value
     *
     * @param string $path
     * @param int|null $storeId
     * @return mixed
     */
    public function conf($path, $storeId = null)
    {
        return $this->scopeConfig->getValue($path, ScopeInterface::SCOPE_STORE, $storeId);
    }

    /**
     * Set config value
     *
     * @param string $path
     * @param mixed $value
     * @return void
     */
    public function setConfig($path, $value)
    {
        $this->configWriter->save($path, $value);
        $this->flushConfigCache();
    }

    /**
     * Flush config cache
     *
     * @return void
     */
    public function flushConfigCache()
    {
        $this->cacheManager->clean(['config', 'block_html']);
    }

    /**
     * Get API settings
     *
     * @param string $path
     * @return mixed
     */
    public function getApiSettings($path)
    {
        return $this->conf(self::API_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Return field value or array of field values
     *
     * @param string|null $path
     * @return array|string
     */
    public function getGeneralSettings($path = null)
    {
        $path = $path ? '/' . $path : '';
        return $this->conf(self::GENERAL_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get attributes settings
     *
     * @param string $path
     * @return string
     */
    public function getAttributesSettings($path)
    {
        return $this->conf(self::ATTRIBUTES_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get description settings
     *
     * @param string $path
     * @return int
     */
    public function getDescriptionSettings($path)
    {
        return (int) $this->conf(self::DESCRIPTION_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get excerpt settings
     *
     * @param string $path
     * @return int
     */
    public function getExcerptSettings($path)
    {
        return (int) $this->conf(self::EXCERPT_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get description settings
     *
     * @param string $path
     * @return int
     */
    public function getCategoryDescriptionSettings($path)
    {
        return (int) $this->conf(self::CATEGORY_DESCRIPTION_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get keyword optimization settings
     *
     * @param string $path
     * @return mixed
     */
    public function getKeywordOptimizationSettings($path = null)
    {
        $path = $path ? '/' . $path : '';
        return $this->conf(self::KEYWORD_OPTIMIZATION_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Return field value or array of field values
     *
     * @param string|null $path
     * @return array|string
     */
    public function getMappingSettings($path = null)
    {
        if (!$path && !empty($this->mappingSettings)) {
            return $this->mappingSettings;
        }
        
        $path = $path ? '/' . $path : '';
        $settings = $this->conf(self::MAPPING_SETTINGS_CONFIG_PATH . $path);

        if (!$path) {
            if (!$settings['custom_open_graph']) {
                unset($settings['open_graph']);
            }

            unset($settings['custom_open_graph']);

            $this->mappingSettings = $settings;
        }

        return $settings;
    }

    /**
     * Get cron settings
     *
     * @param string $path
     * @return string
     */
    public function getCronSettings($path = null)
    {
        return $this->conf(self::CRON_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get setup settings
     *
     * @param string $path
     * @return string
     */
    public function getSetupSettings($path = null)
    {
        return $this->conf(self::SETUP_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get max length settings
     *
     * @param string $path
     * @return string
     */
    public function getMaxLengthSettings($path = null)
    {
        return $this->conf(self::MAX_LENGTH_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get translation settings
     *
     * @param string $path
     * @return string
     */
    public function getTranslationSettings($path = null)
    {
        return $this->conf(self::TRANSLATION_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get dev settings
     *
     * @param string $path
     * @return string
     */
    public function getDevSettings($path = null)
    {
        return $this->conf(self::DEV_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get cannibalization settings
     *
     * @param string $path
     * @return string
     */
    public function getCannibalizationSettings($path = null)
    {
        return $this->conf(self::CANNIBALIZATION_SETTINGS_CONFIG_PATH . $path);
    }

    /**
     * Get autotransfer settings
     *
     * @param string $path
     * @return string
     */
    public function getAutotransferSettings($path = null)
    {
        return $this->conf(self::AUTOTRANSFER_SETTINGS_CONFIG_PATH . $path);
    }
}
