/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

define([
    'ko',
    'WriteTextAI_WriteTextAI/js/model/grid/listing',
    'domReady!'
], function (ko, listing) {
    'use strict';

    var commentHidden = ko.observable(false),
        statuses = ko.observableArray([]);

    /**
     * Add or update a status based on its ID
     *
     * @param {Object} newStatus - The status object to add or update
     */
    var addOrUpdateStatus = function (newStatus) {
        var existingStatuses = statuses();
        var existingIndex = existingStatuses.findIndex(function (status) {
            return status.id === newStatus.id;
        });
        
        if (existingIndex !== -1) {
            /** Update existing status */
            existingStatuses[existingIndex] = newStatus;
            statuses(existingStatuses);
        } else {
            /** Add new status */
            statuses.push(newStatus);
        }
    };
    
    /**
     * Get a status by storeId, recordId, and entityType
     *
     * @param {number} storeId - The store ID
     * @param {string|number} recordId - The record ID
     * @param {string} entityType - The entity type (Product or Category)
     * @returns {Object|undefined} The matching status object or undefined
     */
    var getStatusByRecord = function (storeId, recordId, entityType) {
        var currentUser = listing.currentUser();
        
        return statuses().find(function (status) {
            /** Check if status belongs to current user */
            if (status.userName && status.userName !== currentUser.email) {
                return false;
            }
            
            if (status.queuedIds && status.queuedIds.length > 0) {
                return status.queuedIds.some(function (queuedId) {
                    /** Format: Product_{store id}_{language}_{record id} or Category_{store id}_{language}_{record id} */
                    var parts = queuedId.split('_');
                    if (parts.length >= 4) {
                        var queuedEntityType = parts[0];
                        var queuedStoreId = parts[1];
                        var queuedRecordId = parts[3];
                        return queuedEntityType === entityType &&
                               Number(queuedStoreId) === Number(storeId) &&
                               queuedRecordId === String(recordId);
                    }
                    return false;
                });
            }
        });
    };
    
    /**
     * Get a status by storeId, recordId, and entityType
     *
     * @param {number} storeId - The store ID
     * @param {string|number} recordId - The record ID
     * @param {string} entityType - The entity type (Product or Category)
     * @returns {Object|undefined} The matching status object or undefined
     */
    var getStatusByRecordNormalId = function (storeId, recordId, entityType) {
        var currentUser = listing.currentUser();
        
        return statuses().find(function (status) {
            /** Check if status belongs to current user */
            if (status.userName && status.userName !== currentUser.email) {
                return false;
            }
            
            if (status.queuedIds && status.queuedIds.length > 0) {
                return status.queuedIds.some(function (queuedId) {
                    return queuedId === String(recordId)
                        && status.recordType === entityType
                        && Number(status.storeId) === Number(storeId);
                });
            }
        });
    };

    return {
        commentHidden: commentHidden,
        statuses: statuses,
        addOrUpdateStatus: addOrUpdateStatus,
        getStatusByRecord: getStatusByRecord,
        getStatusByRecordNormalId: getStatusByRecordNormalId
    };
});
