<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\Data\UserWebTokenInterface;
use WriteTextAI\WriteTextAI\Api\Data\UserWebTokenInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\UserWebTokenSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\UserWebTokenRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ResourceModel\UserWebToken as ResourceUserWebToken;
use WriteTextAI\WriteTextAI\Model\ResourceModel\UserWebToken\CollectionFactory as UserWebTokenCollectionFactory;

class UserWebTokenRepository implements UserWebTokenRepositoryInterface
{
    /**
     * @var UserWebToken
     */
    protected $searchResultsFactory;

    /**
     * @var ResourceUserWebToken
     */
    protected $resource;

    /**
     * @var UserWebTokenInterfaceFactory
     */
    protected $userWebTokenFactory;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var UserWebTokenCollectionFactory
     */
    protected $userWebTokenCollectionFactory;

    /**
     * @param ResourceUserWebToken $resource
     * @param UserWebTokenInterfaceFactory $userWebTokenFactory
     * @param UserWebTokenCollectionFactory $userWebTokenCollectionFactory
     * @param UserWebTokenSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceUserWebToken $resource,
        UserWebTokenInterfaceFactory $userWebTokenFactory,
        UserWebTokenCollectionFactory $userWebTokenCollectionFactory,
        UserWebTokenSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->userWebTokenFactory = $userWebTokenFactory;
        $this->userWebTokenCollectionFactory = $userWebTokenCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(UserWebTokenInterface $userWebToken)
    {
        try {
            $this->resource->save($userWebToken);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the userWebToken: %1',
                $exception->getMessage()
            ));
        }
        return $userWebToken;
    }

    /**
     * @inheritDoc
     */
    public function get($userWebTokenId)
    {
        $userWebToken = $this->userWebTokenFactory->create();
        $this->resource->load($userWebToken, $userWebTokenId);
        if (!$userWebToken->getId()) {
            throw new NoSuchEntityException(__('UserWebToken with id "%1" does not exist.', $userWebTokenId));
        }
        return $userWebToken;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->userWebTokenCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(UserWebTokenInterface $userWebToken)
    {
        try {
            $userWebTokenModel = $this->userWebTokenFactory->create();
            $this->resource->load($userWebTokenModel, $userWebToken->getUserwebtokenId());
            $this->resource->delete($userWebTokenModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the UserWebToken: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($userWebTokenId)
    {
        return $this->delete($this->get($userWebTokenId));
    }
}
