<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

 declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\Data\UserFeedbackInterface;
use WriteTextAI\WriteTextAI\Api\Data\UserFeedbackInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\UserFeedbackSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\UserFeedbackRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ResourceModel\UserFeedback as ResourceUserFeedback;
use WriteTextAI\WriteTextAI\Model\ResourceModel\UserFeedback\CollectionFactory as UserFeedbackCollectionFactory;

class UserFeedbackRepository implements UserFeedbackRepositoryInterface
{

    /**
     * @var UserFeedbackCollectionFactory
     */
    protected $userFeedbackCollectionFactory;

    /**
     * @var UserFeedbackInterfaceFactory
     */
    protected $userFeedbackFactory;

    /**
     * @var ResourceUserFeedback
     */
    protected $resource;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var UserFeedback
     */
    protected $searchResultsFactory;

    /**
     * @param ResourceUserFeedback $resource
     * @param UserFeedbackInterfaceFactory $userFeedbackFactory
     * @param UserFeedbackCollectionFactory $userFeedbackCollectionFactory
     * @param UserFeedbackSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceUserFeedback $resource,
        UserFeedbackInterfaceFactory $userFeedbackFactory,
        UserFeedbackCollectionFactory $userFeedbackCollectionFactory,
        UserFeedbackSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->userFeedbackFactory = $userFeedbackFactory;
        $this->userFeedbackCollectionFactory = $userFeedbackCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(UserFeedbackInterface $userFeedback)
    {
        try {
            $this->resource->save($userFeedback);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the userFeedback: %1',
                $exception->getMessage()
            ));
        }
        return $userFeedback;
    }

    /**
     * @inheritDoc
     */
    public function get($userFeedbackId)
    {
        $userFeedback = $this->userFeedbackFactory->create();
        $this->resource->load($userFeedback, $userFeedbackId);
        if (!$userFeedback->getId()) {
            throw new NoSuchEntityException(__('user_feedback with id "%1" does not exist.', $userFeedbackId));
        }
        return $userFeedback;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->userFeedbackCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(UserFeedbackInterface $userFeedback)
    {
        try {
            $userFeedbackModel = $this->userFeedbackFactory->create();
            $this->resource->load($userFeedbackModel, $userFeedback->getUserFeedbackId());
            $this->resource->delete($userFeedbackModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the user_feedback: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($userFeedbackId)
    {
        return $this->delete($this->get($userFeedbackId));
    }
}
