<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Model\AbstractModel;
use WriteTextAI\WriteTextAI\Model\ApiManager;
use WriteTextAI\WriteTextAI\Helper\IntegrationTokenHelper;
use Magento\Store\Model\StoreManagerInterface;

class IntegrationApiManager extends AbstractModel
{
    /**
     * API Manager
     *
     * @var ApiManager
     */
    protected $apiManager;

    /**
     * Integration Token Helper
     *
     * @var IntegrationTokenHelper
     */
    protected $integrationTokenHelper;

    /**
     * Store Manager
     *
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * Constructor
     *
     * @param ApiManager $apiManager
     * @param IntegrationTokenHelper $integrationTokenHelper
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        ApiManager $apiManager,
        IntegrationTokenHelper $integrationTokenHelper,
        StoreManagerInterface $storeManager
    ) {
        $this->apiManager = $apiManager;
        $this->integrationTokenHelper = $integrationTokenHelper;
        $this->storeManager = $storeManager;
    }

    /**
     * Send API Integration
     */
    public function sendApiIntegration()
    {
        try {
            $integrationData = $this->integrationTokenHelper->getIntegrationData();
            if (!$integrationData) {
                return false;
            }
            return $this->sendToApi($integrationData);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Check if integration is still valid
     *
     * @return bool
     */
    public function checkIfIntegrationIsStillValid()
    {
        try {
            $integrationData = $this->integrationTokenHelper->getIntegrationByName();
            if (!$integrationData) {
                $integrationData = $this->integrationTokenHelper->createIntegration();
                $this->sendToApi($integrationData);
            }
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Send to API
     *
     * @param array $integrationData
     * @return bool
     */
    public function sendToApi($integrationData)
    {
        $body = [];
            
        $baseUrl = $this->storeManager->getStore()->getBaseUrl(
            \Magento\Framework\UrlInterface::URL_TYPE_DIRECT_LINK
        );
        $body['restBaseUrl'] = $baseUrl . 'rest/V1/writetextai/';
        $body['authenticationType'] = 'OAuth1.0';
        $body['authenticationParameters'] = [
            "consumerKey" => $integrationData['consumer_key'],
            "consumerSecret" => $integrationData['consumer_secret'],
            "accessToken" => $integrationData['access_token'],
            "accessTokenSecret" => $integrationData['access_token_secret'],
            "signature" => 'HMAC-SHA256',
        ];
        return $this->apiManager->sendApiIntegration($body);
    }
}
