<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Helper;

use WriteTextAI\WriteTextAI\Helper\Settings;
use WriteTextAI\WriteTextAI\Model\OptionSource\Filter\Fields;
use WriteTextAI\WriteTextAI\Model\Config\Source\TriggerActions;
use WriteTextAI\WriteTextAI\Model\Config\Source\AutomaticTextOptimization;

class Optimization extends Settings
{
    /**
     * Get global triggers
     *
     * @param string $entityType
     *
     * @return array
     */
    public function getGlobalTriggers($entityType)
    {
        $difficultyEquivalents = [
            'low' => 'Low',
            'low_mid' => 'LowMid',
            'mid' => 'Mid',
            'mid_high' => 'MidHigh'
        ];
        $difficulties = array_keys($difficultyEquivalents);

        $triggers = [];
        foreach ($difficulties as $difficulty) {
            $triggers[$difficultyEquivalents[$difficulty]] = [
                'option' => $this->getTrigger($difficulty),
                'value' => $this->getTriggerValue($difficulty),
                'action' => $this->getAction(),
                'text_types' => $this->getTextTypes('Product')

            ];
        }

        return $triggers;
    }

    /**
     * Get action
     *
     * @return string
     */
    protected function getAction()
    {
        $keywordOptimization = $this->getKeywordOptimizationSettings();
        $actionsEquivalent = [
            AutomaticTextOptimization::AUTO_REWRITE_AND_QUEUE => 'AutomaticRewrite',
            AutomaticTextOptimization::FLAG => 'FlagForRewrite',
            AutomaticTextOptimization::AUTO_REWRITE_AND_TRANSFER => 'FullAutomation'
        ];
        $optionValue = $keywordOptimization['automatic_text_optimization'] ?? null;

        return $actionsEquivalent[$optionValue] ?? null;
    }

    /**
     * Get trigger
     *
     * @param string $difficulty
     *
     * @return string
     */
    protected function getTrigger($difficulty)
    {
        $keywordOptimization = $this->getKeywordOptimizationSettings();
        $optionsEquivalent = [
            TriggerActions::WHEN_PAGE_IS_RANKING => 'WhenRanking',
            TriggerActions::AFTER_N_DAYS => 'AfterSetDays',
            TriggerActions::NONE => 'None'
        ];
        $optionValue = $keywordOptimization['pipeline_keyword'][$difficulty]['trigger_action'] ?? null;

        return $optionsEquivalent[$optionValue] ?? null;
    }

    /**
     * Get trigger value
     *
     * @param string $difficulty
     *
     * @return string
     */
    protected function getTriggerValue($difficulty)
    {
        $keywordOptimization = $this->getKeywordOptimizationSettings();
        $valueReference = [
            TriggerActions::WHEN_PAGE_IS_RANKING => 'trigger_action_n_position',
            TriggerActions::AFTER_N_DAYS => 'trigger_action_n_days',
            TriggerActions::NONE => 'trigger_action_none'
        ];
        $trigger = $keywordOptimization['pipeline_keyword'][$difficulty]['trigger_action'];

        return $keywordOptimization['pipeline_keyword'][$difficulty][$valueReference[$trigger]] ?? null;
    }

    /**
     * Get text types
     *
     * @param string $entityType
     *
     * @return array
     */
    protected function getTextTypes($entityType)
    {
        $keywordOptimization = $this->getKeywordOptimizationSettings();
        $textTypes = [];
        switch ($entityType) {
            case 'Product':
                $selectedProductsTextType = explode(',', $keywordOptimization['products'] ?? '');
                $productFieldsMapping = [
                    'page_title' => Fields::PAGE_TITLE,
                    'page_description' => Fields::PAGE_DESCRIPTION,
                    'product_description' => Fields::PRODUCT_DESCRIPTION,
                    'short_product_description' => Fields::EXCERPT,
                    'open_graph' => Fields::OPEN_GRAPH
                ];
                foreach ($selectedProductsTextType as $textType) {
                    $textTypes[] = $productFieldsMapping[$textType];
                }
                return $textTypes;
            case 'Category':
                $selectedCategoryTextType = explode(',', $keywordOptimization['categories'] ?? '');
                $categoryFieldsMapping = Fields::CATEGORY_MAPPING;
                foreach ($selectedCategoryTextType as $textType) {
                    $textTypes[] = $categoryFieldsMapping[$textType];
                }
                return $textTypes;
            default:
                return [];
        }
    }
}
