<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Controller\Adminhtml\Grid;

use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Controller\ResultFactory;
use WriteTextAI\WriteTextAI\Model\BulkTransferRequestsManager;
use WriteTextAI\WriteTextAI\Model\BulkGenerateRequestsManager;
use WriteTextAI\WriteTextAI\Model\ApiManager;
use WriteTextAI\WriteTextAI\Helper\Store as StoreHelper;
use Magento\Cron\Model\ScheduleFactory;
use Magento\Cron\Model\ResourceModel\Schedule as ScheduleResource;
use Magento\Framework\Stdlib\DateTime\DateTime;

class TransferViaSchedule extends Action implements HttpPostActionInterface
{
    public const ADMIN_RESOURCE = 'WriteTextAI_WriteTextAI::transfer';

    /**
     * @var BulkTransferRequestsManager
     */
    protected $bulkTransferRequestsManager;
    
    /**
     * @var BulkGenerateRequestsManager
     */
    protected $bulkGenerateRequestsManager;
    
    /**
     * @var ApiManager
     */
    protected $apiManager;
    
    /**
     * @var StoreHelper
     */
    protected $storeHelper;
    
    /**
     * @var ScheduleFactory
     */
    protected $scheduleFactory;
    
    /**
     * @var ScheduleResource
     */
    protected $scheduleResource;
    
    /**
     * @var DateTime
     */
    protected $dateTime;
    
    /**
     * Constructor
     *
     * @param Context $context
     * @param BulkTransferRequestsManager $bulkTransferRequestsManager
     * @param BulkGenerateRequestsManager $bulkGenerateRequestsManager
     * @param ApiManager $apiManager
     * @param StoreHelper $storeHelper
     * @param ScheduleFactory $scheduleFactory
     * @param ScheduleResource $scheduleResource
     * @param DateTime $dateTime
     */
    public function __construct(
        Context $context,
        BulkTransferRequestsManager $bulkTransferRequestsManager,
        BulkGenerateRequestsManager $bulkGenerateRequestsManager,
        ApiManager $apiManager,
        StoreHelper $storeHelper,
        ScheduleFactory $scheduleFactory,
        ScheduleResource $scheduleResource,
        DateTime $dateTime
    ) {
        parent::__construct($context);
        $this->bulkTransferRequestsManager = $bulkTransferRequestsManager;
        $this->bulkGenerateRequestsManager = $bulkGenerateRequestsManager;
        $this->apiManager = $apiManager;
        $this->storeHelper = $storeHelper;
        $this->scheduleFactory = $scheduleFactory;
        $this->scheduleResource = $scheduleResource;
        $this->dateTime = $dateTime;
    }

    /**
     * Execute
     *
     * @return \Magento\Framework\Controller\Result\Json
     */
    public function execute()
    {
        $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        try {
            $params = $this->getRequest()->getParams();
            if (isset($params['is_cancel']) && $params['is_cancel'] === 'true') {
                $this->bulkTransferRequestsManager->delete($params['user']);
            } else {
                $savedId = $this->bulkTransferRequestsManager->saveBulkRequest(
                    $params['user'],
                    $params['stores'],
                    $params['store_id'],
                    $params['fields'],
                    $params['queue_ids'] ?? [],
                    $params['completed_ids'] ?? []
                );
                $this->bulkGenerateRequestsManager->deleteByUser($params['user']);
                
                // Send initial "Running" notification
                $this->sendBulkTransferInitialNotification(
                    $savedId,
                    $params['user'],
                    $params['store_id'],
                    $params['queue_ids'] ?? []
                );
                
                // Schedule the grid transfer cron job to run immediately
                $this->scheduleGridTransferCron($savedId);
            }

            $result = [
                'success' => true
            ];
        } catch (\Exception $e) {
            $message = $e->getMessage();
            if (!$message) {
                $message = __(
                    'A system error has occurred. Please try again. If the issue persists,'
                    . ' please contact our support team at support@writetext.ai.'
                );
            }
            
            $result = [
                'success' => false,
                'message' => $message
            ];
        }

        $response->setData($result);

        return $response;
    }
    
    /**
     * Schedule grid transfer cron job to run on next cron execution
     *
     * @param int $bulkRequestId
     * @return void
     */
    private function scheduleGridTransferCron($bulkRequestId): void
    {
        try {
            // Find an available job code from wtai_grid_transfer to wtai_grid_transfer_20
            $availableJobCode = $this->findAvailableGridTransferJobCode();
            
            if ($availableJobCode) {
                $schedule = $this->scheduleFactory->create();
                
                // Schedule at the next minute mark using Magento's default pattern
                $currentTime = $this->dateTime->gmtTimestamp();
                $currentSeconds = (int)date('s', $currentTime);
                
                // If we're close to the next minute (46+ seconds), skip to the minute after next
                // This provides buffer time to ensure the cron job has time to start properly
                if ($currentSeconds >= 46) {
                    $nextMinute = $currentTime + 120; // Add 2 minutes (skip next minute, schedule for the one after)
                } else {
                    $nextMinute = $currentTime + 60; // Add 60 seconds to get next minute
                }
                
                $scheduledAt = date('Y-m-d H:i:00', $nextMinute); // Magento's format with :00 seconds
                
                $schedule->setJobCode($availableJobCode)
                    ->setStatus(\Magento\Cron\Model\Schedule::STATUS_PENDING)
                    ->setCreatedAt($this->dateTime->gmtDate())
                    ->setScheduledAt($scheduledAt)
                    ->setMessages('bulk_request_id:' . $bulkRequestId);
                $this->scheduleResource->save($schedule);
            }
            // If no available job code, the request will still be saved but won't be scheduled
            // This prevents overloading the system with too many concurrent jobs
        } catch (\Exception $e) {
            // Log error but don't break the main functionality
            // The transfer request will still be saved even if cron scheduling fails
        }
    }
    
    /**
     * Find an available grid transfer job code that's not currently running or pending
     *
     * @return string|null
     */
    private function findAvailableGridTransferJobCode(): ?string
    {
        try {
            $connection = $this->scheduleResource->getConnection();
            $tableName = $this->scheduleResource->getMainTable();
            
            // Get all currently running or pending job codes for grid transfer jobs
            $select = $connection->select()
                ->from($tableName, ['job_code'])
                ->where('job_code LIKE ?', 'wtai_grid_transfer%')
                ->where('status IN (?)', [
                    \Magento\Cron\Model\Schedule::STATUS_PENDING,
                    \Magento\Cron\Model\Schedule::STATUS_RUNNING
                ]);
                
            $busyJobCodes = $connection->fetchCol($select);
            
            // Check each job code from wtai_grid_transfer to wtai_grid_transfer_20
            $jobCodes = ['wtai_grid_transfer']; // Start with base job code
            for ($i = 1; $i <= 20; $i++) {
                $jobCodes[] = 'wtai_grid_transfer_' . $i;
            }
            
            // Find the first available job code
            foreach ($jobCodes as $jobCode) {
                if (!in_array($jobCode, $busyJobCodes)) {
                    return $jobCode;
                }
            }
            
            return null; // All job codes are busy
        } catch (\Exception $e) {
            // If there's an error, fallback to the original job code
            return 'wtai_grid_transfer';
        }
    }
    
    /**
     * Send initial bulk transfer notification when request is created
     *
     * @param int $bulkRequestId
     * @param string $userName
     * @param int $storeFilter
     * @param array $queueIds
     * @return void
     */
    private function sendBulkTransferInitialNotification(
        int $bulkRequestId,
        string $userName,
        int $storeFilter,
        array $queueIds
    ): void {
        try {
            $language = $this->storeHelper->getFormattedLanguage($storeFilter);
            $total = count($queueIds);
            
            $notificationData = [
                'type' => "8",
                'subType' => 'transfer',
                'id' => $bulkRequestId,
                'userName' => $userName,
                'recordType' => 'Product',
                'storeId' => (string) $storeFilter,
                'status' => 'Pending',
                'startTime' => $this->dateTime->gmtDate(),
                'endTime' => $this->dateTime->gmtDate(),
                'queuedIds' => $queueIds,
                'runningIds' => [],
                'completedIds' => [],
                'failedIds' => [],
                'completed' => 0,
                'failed' => 0,
                'total' => $total,
                'language' => $language,
                'metaData' => []
            ];
            
            // Send notification with single retry
            try {
                $this->apiManager->sendCustomNotification($notificationData);
            } catch (\Exception $retryException) {
                // Single retry attempt
                $this->apiManager->sendCustomNotification($notificationData);
            }
        } catch (\Exception $e) {
            // Don't throw exception - notification failure shouldn't stop the process
            // Could add logging here if needed
        }
    }
}
