<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\ViewModel;

use Magento\Framework\View\LayoutInterface;
use WriteTextAI\WriteTextAI\Helper\Data;
use WriteTextAI\WriteTextAI\Model\Config\Source\CategoryTexts;
use WriteTextAI\WriteTextAI\Model\Config\Source\ProductTexts;
use WriteTextAI\WriteTextAI\Model\SettingManager;
use WriteTextAI\WriteTextAI\Model\Config\Source\AutomaticTextOptimization;
use WriteTextAI\WriteTextAI\Model\OptionSource\Filter\Fields;

class PipelineSettings implements \Magento\Framework\View\Element\Block\ArgumentInterface
{
    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var ProductTexts
     */
    protected $productTexts;

    /**
     * @var CategoryTexts
     */
    protected $categoryTexts;

    /**
     * @var SettingManager
     */
    protected $settingManager;

    /**
     * Constructor
     *
     * @param Data $helper
     * @param ProductTexts $productTexts
     * @param CategoryTexts $categoryTexts
     * @param SettingManager $settingManager
     */
    public function __construct(
        Data $helper,
        ProductTexts $productTexts,
        CategoryTexts $categoryTexts,
        SettingManager $settingManager
    ) {
        $this->helper = $helper;
        $this->productTexts = $productTexts;
        $this->categoryTexts = $categoryTexts;
        $this->settingManager = $settingManager;
    }
    
    /**
     * Get automatic text optimization
     *
     * @return array
     */
    public function getAutomaticTextOptimization()
    {
        return $this->helper->getAutomaticTextOptimizationOptions();
    }

    /**
     * Get product fields
     *
     * @return array
     */
    public function getProductFields()
    {
        return $this->productTexts->toOptionArray();
    }

    /**
     * Get text fields
     *
     * @param string $entityType
     *
     * @return array
     */
    public function getTriggerSettingsTextFields($entityType)
    {
        if ($entityType == 'Product') {
            return $this->productTexts->toOptionArray();
        } else {
            return $this->categoryTexts->toOptionArray();
        }
    }

    /**
     * Get rules
     *
     * @return array
     */
    public function getRules()
    {
        return $this->settingManager->getSettings('rules');
    }

    /**
     * Get default automatic text optimization
     *
     * @return string
     */
    public function getDefaultAutomaticTextOptimization()
    {
        /**
         * Add default logic
         */

        return AutomaticTextOptimization::AUTO_REWRITE_AND_QUEUE;
    }

    /**
     * Get trigger action n position default value
     *
     * @return string
     */
    public function getTriggerActionNPositionDefaultValue()
    {
        /**
         * Add default logic
         */

        $rules = $this->getRules();
        return $rules['minTriggerActionNPosition'] ?? 2;
    }

    /**
     * Get trigger action after n days default value
     *
     * @return string
     */
    public function getTriggerActionNDaysDefaultValue()
    {
        /**
         * Add default logic
         */
        
        $rules = $this->getRules();
        return $rules['minTriggerActionNDays'] ?? 1;
    }

    /**
     * Get trigger action n position default value per difficulty
     *
     * @param string $difficulty
     *
     * @return string
     */
    public function getTriggerActionNPositionDefaultValuePerDifficulty($difficulty)
    {
        $settingDifficulty = $difficultySettingsEquivalent[$difficulty] ?? 'low';
        $defaultVal = $this->helper->getKeywordOptimizationSettings('pipeline_keyword');
        if (isset($defaultVal[$difficulty]['trigger_action_n_position'])) {
            return $defaultVal[$difficulty]['trigger_action_n_position'];
        }
        $rules = $this->getRules();
        return $rules['minTriggerActionNPositionPerDifficulty'] ?? 1;
    }
    
    /**
     * Get trigger action after n days default value per difficulty
     *
     * @param string $difficulty
     *
     * @return string
     */
    public function getTriggerActionNDaysDefaultValuePerDifficulty($difficulty)
    {
        $settingDifficulty = $difficultySettingsEquivalent[$difficulty] ?? 'low';
        $defaultVal = $this->helper->getKeywordOptimizationSettings('pipeline_keyword');
        if (isset($defaultVal[$difficulty]['trigger_action_n_days'])) {
            return $defaultVal[$difficulty]['trigger_action_n_days'];
        }
        $rules = $this->getRules();
        return $rules['minTriggerActionNDaysPerDifficulty'] ?? 30;
    }

    /**
     * Get text fields mapping
     *
     * @param string $entityType
     *
     * @return array
     */
    public function getTextFieldsMapping($entityType)
    {
        if ($entityType == 'Product') {
            $textFieldMapping = [
                'page_title' => Fields::PAGE_TITLE,
                'page_description' => Fields::PAGE_DESCRIPTION,
                'product_description' => Fields::PRODUCT_DESCRIPTION,
                'short_product_description' => Fields::EXCERPT
            ];
            if ($this->isCustomOpenGraphEnabled()) {
                $textFieldMapping['open_graph'] = Fields::OPEN_GRAPH;
            }
            return $textFieldMapping;
        } else {
            return Fields::CATEGORY_MAPPING;
        }
    }

    /**
     * Get custom open graph
     *
     * @return bool
     */
    public function isCustomOpenGraphEnabled()
    {
        return $this->helper->getCustomOpenGraph();
    }
}
