<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */
declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\Data\SettingInterface;
use WriteTextAI\WriteTextAI\Api\Data\SettingInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\SettingSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\SettingRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ResourceModel\Setting as ResourceSetting;
use WriteTextAI\WriteTextAI\Model\ResourceModel\Setting\CollectionFactory as SettingCollectionFactory;

class SettingRepository implements SettingRepositoryInterface
{
    /**
     * @var Setting
     */
    protected $searchResultsFactory;

    /**
     * @var SettingInterfaceFactory
     */
    protected $settingFactory;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var SettingCollectionFactory
     */
    protected $settingCollectionFactory;

    /**
     * @var ResourceSetting
     */
    protected $resource;

    /**
     * @param ResourceSetting $resource
     * @param SettingInterfaceFactory $settingFactory
     * @param SettingCollectionFactory $settingCollectionFactory
     * @param SettingSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceSetting $resource,
        SettingInterfaceFactory $settingFactory,
        SettingCollectionFactory $settingCollectionFactory,
        SettingSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->settingFactory = $settingFactory;
        $this->settingCollectionFactory = $settingCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(SettingInterface $setting)
    {
        try {
            $this->resource->save($setting);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the setting: %1',
                $exception->getMessage()
            ));
        }
        return $setting;
    }

    /**
     * @inheritDoc
     */
    public function get($settingId)
    {
        $setting = $this->settingFactory->create();
        $this->resource->load($setting, $settingId);
        if (!$setting->getId()) {
            throw new NoSuchEntityException(__('Setting with id "%1" does not exist.', $settingId));
        }
        return $setting;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->settingCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(SettingInterface $setting)
    {
        try {
            $settingModel = $this->settingFactory->create();
            $this->resource->load($settingModel, $setting->getSettingId());
            $this->resource->delete($settingModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the Setting: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($settingId)
    {
        return $this->delete($this->get($settingId));
    }
}
