<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Model;

use WriteTextAI\WriteTextAI\Model\GeneratedText;
use WriteTextAI\WriteTextAI\ViewModel\Fields;
use WriteTextAI\WriteTextAI\Helper\Product as ProductHelper;
use WriteTextAI\WriteTextAI\Helper\Store as StoreHelper;

class MarkReview
{
    /**
     * @var GeneratedText
     */
    protected $generatedText;

    /**
     * @var Fields
     */
    protected $fields;

    /**
     * @var ProductHelper
     */
    protected $productHelper;

    /**
     * @var StoreHelper
     */
    protected $storeHelper;

    /**
     * Constructor
     *
     * @param GeneratedText $generatedText
     * @param Fields $fields
     * @param ProductHelper $productHelper
     * @param StoreHelper $storeHelper
     */
    public function __construct(
        GeneratedText $generatedText,
        Fields $fields,
        ProductHelper $productHelper,
        StoreHelper $storeHelper
    ) {
        $this->generatedText = $generatedText;
        $this->fields = $fields;
        $this->productHelper = $productHelper;
        $this->storeHelper = $storeHelper;
    }

    /**
     * Get reviewed
     *
     * @param int $productId
     * @param int $storeId
     * @param array $text
     * @return bool
     */
    public function getReviewed($productId, $storeId, $text = [])
    {
        $textsReviewed = $this->getTextsReview($productId, $storeId, $text);
        $imagesReviewed = $this->getImagesReview($productId, $storeId);

        return $textsReviewed && $imagesReviewed;
    }

    /**
     * Get texts review
     *
     * @param int $productId
     * @param int $storeId
     * @param array $text
     * @return bool
     */
    private function getTextsReview($productId, $storeId, $text = [])
    {
        $text = !empty($text) ? $text : $this->generatedText->getGenerated($productId, $storeId);

        if (!isset($text['records']) || empty($text['records'])) {
            return false;
        }

        $fieldsReviewed = [];
        foreach ($text['records'] as $record) {
            if ($record['recordId'] == $productId && isset($record['stores'])) {
                foreach ($record['stores'] as $store) {
                    if ($store['storeId'] == $storeId && (
                        $store['field'] !== 'open graph text' || (
                            $store['field'] === 'open graph text' &&
                            $this->fields->getCustomOpenGraph()
                        )
                    )) {
                        if (isset($store['texts'][0]['history'][0]['reviewed'])) {
                            $fieldsReviewed[] = $store['texts'][0]['history'][0]['reviewed'];
                        } else {
                            $fieldsReviewed[] = false;
                        }
                    }
                }
            }
        }

        return !empty($fieldsReviewed) && in_array(true, $fieldsReviewed);
    }

    /**
     * Get images review
     *
     * @param int $productId
     * @param int $storeId
     *
     * @return void
     */
    private function getImagesReview($productId, $storeId)
    {
        $language = $this->storeHelper->getFormattedLanguage($storeId);
        $product = $this->productHelper->getProduct($productId, $storeId);
        $images = $this->productHelper->getImages($product, $storeId, $language);

        if (empty($images)) {
            return true;
        }
        
        $reviews = [];
        foreach ($images as $image) {
            if (empty($image['writetext_alt_id'])) {
                continue;
            }
            $reviews[] = $image['reviewed'];
        }

        return !empty($reviews) && in_array(true, $reviews);
    }
}
