<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Helper;

use WriteTextAI\WriteTextAI\Helper\Data;
use Magento\Store\Model\StoreManagerInterface;

class Store extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * Constructor
     *
     * @param Data $helper
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        Data $helper,
        StoreManagerInterface $storeManager
    ) {
        $this->helper = $helper;
        $this->storeManager = $storeManager;
    }

    /**
     * Get frontend store id
     *
     * @param int $storeId
     * @return int
     */
    public function getFrontendStoreId($storeId)
    {
        if ($storeId == \Magento\Store\Model\Store::DEFAULT_STORE_ID) {
            return $this->storeManager->getDefaultStoreView()->getId();
        }

        return $storeId;
    }

    /**
     * Get store scope
     *
     * @param int $storeId
     * @return string
     */
    public function getStoreScope($storeId)
    {
        if ($storeId == \Magento\Store\Model\Store::DEFAULT_STORE_ID) {
            $scope = __('All store views');
        } else {
            $store = $this->storeManager->getStore($storeId);
            $scope = $store->getName();
        }

        return $scope;
    }
    
    /**
     * Get website by store id
     *
     * @param int $storeId
     * @return \Magento\Store\Api\Data\WebsiteInterface
     */
    public function getWebsiteByStoreId($storeId)
    {
        $store = $this->storeManager->getStore($storeId);
        $website = $store->getWebsite();

        return $website;
    }

    /**
     * Get store language
     *
     * @param int $storeId
     * @return string
     */
    public function getStoreLanguage($storeId)
    {
        $locale = $this->helper->conf('general/locale/code', $storeId);

        return $locale;
    }

    /**
     * Get formatted language
     *
     * @param int $storeId
     * @return string
     */
    public function getFormattedLanguage($storeId)
    {
        $language = $this->getStoreLanguage($storeId);
        $language = str_replace('_', '-', $language);

        return $language;
    }

    /**
     * Get region independent language
     *
     * @param int $storeId
     * @return string
     */
    public function getRegionIndependentLanguage($storeId)
    {
        $language = $this->getStoreLanguage($storeId);
        $language = explode('_', (string) $language)[0];

        return $language;
    }

    /**
     * Get formatted region independent language
     *
     * @param int $storeId
     * @return string
     */
    public function getFormattedRegionIndependentLanguage($storeId)
    {
        $language = $this->getFormattedLanguage($storeId);
        $language = explode('-', (string) $language)[0];

        return $language;
    }

    /**
     * Get language code
     *
     * @param int $storeId
     *
     * @return string
     */
    public function getLanguageCode($storeId)
    {
        $languageCode = $this->getRegionIndependentLanguage($storeId);

        return strtoupper($languageCode);
    }

    /**
     * Get all stores
     *
     * @param bool $includeDefault
     * @return array
     */
    public function getAllStores($includeDefault = false)
    {
        $stores = $this->storeManager->getStores($includeDefault);

        return $stores;
    }
}
