<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Helper;

use Magento\Store\Model\StoreManagerInterface;
use WriteTextAI\WriteTextAI\Model\GeneratedText;
use Magento\Store\Model\Store;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use WriteTextAI\WriteTextAI\Helper\Store as StoreHelper;
use WriteTextAI\WriteTextAI\Model\Categories\GeneratedText as CategoriesGeneratedText;

class ReviewStatus extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * @var GeneratedText
     */
    protected $generatedText;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var TimezoneInterface
     */
    protected $timezone;

    /**
     * @var StoreHelper
     */
    protected $storeHelper;

    /**
     * @var CategoriesGeneratedText
     */
    protected $categoriesGeneratedText;

    /**
     * Constructor
     *
     * @param GeneratedText $generatedText
     * @param StoreManagerInterface $storeManager
     * @param TimezoneInterface $timezone
     * @param StoreHelper $storeHelper
     * @param CategoriesGeneratedText $categoriesGeneratedText
     */
    public function __construct(
        GeneratedText $generatedText,
        StoreManagerInterface $storeManager,
        TimezoneInterface $timezone,
        StoreHelper $storeHelper,
        CategoriesGeneratedText $categoriesGeneratedText
    ) {
        $this->generatedText = $generatedText;
        $this->storeManager = $storeManager;
        $this->timezone = $timezone;
        $this->storeHelper = $storeHelper;
        $this->categoriesGeneratedText = $categoriesGeneratedText;
    }

    /**
     * Get review status
     *
     * @param int $categoryId
     * @param int $storeId
     * @return array
     */
    public function getCategoryReviewStatus($categoryId, $storeId)
    {
        $reviews = [];
        if ((int)$storeId !== Store::DEFAULT_STORE_ID) {
            $review = $this->categoriesGeneratedText->getReviewStatus(
                $categoryId,
                $storeId,
                'Category'
            );

            if (!empty($review)) {
                $review = $this->formatDate($review);
                $reviews[] = [
                    'id' => $storeId,
                    'name' => '',
                    'reviews' => $review
                ];
            }
        } else {
            $stores = $this->storeManager->getStores();
            $reviewsPerStore = [];
            foreach ($stores as $store) {
                $language = $this->storeHelper->getFormattedLanguage($store->getId());

                if (!isset($reviewsPerStore[$language])) {
                    $reviewsPerStore[$language] = $this->categoriesGeneratedText->getReviewStatus(
                        $categoryId,
                        $store->getId(),
                        'Category'
                    );
                }
                
                $review = $reviewsPerStore[$language];

                if (!empty($review)) {
                    $review = $this->formatDate($review);
                    $reviews[] = [
                        'id' => $store->getId(),
                        'name' => $store->getName(),
                        'reviews' => $review
                    ];
                }
            }
        }

        return $reviews;
    }

    /**
     * Get review status
     *
     * @param int $productId
     * @param int $storeId
     * @return array
     */
    public function getReviewStatus($productId, $storeId)
    {
        $reviews = [];
        if ((int)$storeId !== Store::DEFAULT_STORE_ID) {
            $review = $this->generatedText->getReviewStatus(
                $productId,
                $storeId
            );

            if (!empty($review)) {
                $review = $this->formatDate($review);
                $reviews[] = [
                    'id' => $storeId,
                    'name' => '',
                    'reviews' => $review
                ];
            }
        } else {
            $stores = $this->storeManager->getStores();
            $reviewsPerStore = [];
            foreach ($stores as $store) {
                $language = $this->storeHelper->getFormattedLanguage($store->getId());

                if (!isset($reviewsPerStore[$store->getId()])) {
                    $reviewsPerStore[$store->getId()] = $this->generatedText->getReviewStatus(
                        $productId,
                        $store->getId()
                    );
                }
                
                $review = $reviewsPerStore[$store->getId()];

                if (!empty($review)) {
                    $review = $this->formatDate($review);
                    $reviews[] = [
                        'id' => $store->getId(),
                        'name' => $store->getName(),
                        'reviews' => $review
                    ];
                }
            }
        }

        return $reviews;
    }

    /**
     * Format date
     *
     * @param array $reviews
     * @return string
     */
    protected function formatDate($reviews)
    {
        if (!is_array($reviews)) {
            return [];
        }

        $formattedReviews = [];
        
        foreach ($reviews as $review) {
            $date = $review['date'] ?? '';

            $formattedDate = $this->timezone->date($date);
            $formattedDate = $formattedDate->format('F j, Y \a\t g:i:s A');

            $review['date'] = $formattedDate;

            $formattedReviews[] = $review;
        }

        return $formattedReviews;
    }
}
