<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Cron;

use WriteTextAI\WriteTextAI\Cron\BaseCron;
use Magento\User\Model\UserFactory;
use WriteTextAI\WriteTextAI\Model\Api\Session as ApiSession;
use Magento\Framework\App\Config\ScopeConfigInterface;
use WriteTextAI\WriteTextAI\Model\BulkTransferRequestsManager;
use WriteTextAI\WriteTextAI\Logger\CronLogger;
use Magento\Cron\Model\Schedule;
use Magento\Cron\Model\ScheduleFactory;
use Magento\Cron\Model\ResourceModel\Schedule as ScheduleResource;
use Magento\Framework\Stdlib\DateTime\DateTime;

/**
 * Stuck Bulk Transfer Reschedule Cron Job
 * Checks for bulk transfer requests that don't have active schedules and reschedules them
 */
class StuckBulkTransferReschedule extends BaseCron
{
    /**
     * @var BulkTransferRequestsManager
     */
    protected $bulkTransferRequestsManager;

    /**
     * @var CronLogger
     */
    protected $cronLogger;

    /**
     * @var ScheduleFactory
     */
    protected $scheduleFactory;

    /**
     * @var ScheduleResource
     */
    protected $scheduleResource;

    /**
     * @var DateTime
     */
    protected $dateTime;

    /**
     * Constructor
     *
     * @param UserFactory $userFactory
     * @param ApiSession $apiSession
     * @param ScopeConfigInterface $scopeConfig
     * @param BulkTransferRequestsManager $bulkTransferRequestsManager
     * @param CronLogger $cronLogger
     * @param ScheduleFactory $scheduleFactory
     * @param ScheduleResource $scheduleResource
     * @param DateTime $dateTime
     */
    public function __construct(
        UserFactory $userFactory,
        ApiSession $apiSession,
        ScopeConfigInterface $scopeConfig,
        BulkTransferRequestsManager $bulkTransferRequestsManager,
        CronLogger $cronLogger,
        ScheduleFactory $scheduleFactory,
        ScheduleResource $scheduleResource,
        DateTime $dateTime
    ) {
        parent::__construct($userFactory, $apiSession, $scopeConfig);
        $this->bulkTransferRequestsManager = $bulkTransferRequestsManager;
        $this->cronLogger = $cronLogger;
        $this->scheduleFactory = $scheduleFactory;
        $this->scheduleResource = $scheduleResource;
        $this->dateTime = $dateTime;
    }

    /**
     * Execute stuck bulk transfer reschedule process
     *
     * @param Schedule $schedule
     * @return void
     */
    public function execute(?Schedule $schedule = null): void
    {
        $this->cronLogger->logRequest('StuckBulkTransferReschedule cron job started');

        try {
            // Get all bulk transfer requests
            $bulkRequests = $this->getBulkTransferRequests();
            
            if (empty($bulkRequests)) {
                $this->cronLogger->logRequest('No bulk transfer requests found');
                return;
            }

            $this->cronLogger->logRequest('Found ' . count($bulkRequests) . ' bulk transfer requests to check');

            foreach ($bulkRequests as $bulkRequest) {
                $bulkRequestId = (int)$bulkRequest->getId();
                
                // Check if there's a pending or running schedule for this request
                if (!$this->hasActiveSchedule($bulkRequestId)) {
                    $this->cronLogger->logRequest('No active schedule found for bulk request ID: ' . $bulkRequestId . ', creating new schedule');
                    $this->scheduleGridTransferCron($bulkRequestId);
                } else {
                    $this->cronLogger->logRequest('Active schedule already exists for bulk request ID: ' . $bulkRequestId);
                }
            }

        } catch (\Exception $e) {
            $this->cronLogger->logRequest('Error in StuckBulkTransferReschedule cron: ' . $e->getMessage());
        }

        $this->cronLogger->logRequest('StuckBulkTransferReschedule cron job completed');
    }

    /**
     * Get all bulk transfer requests from the database
     *
     * @return array
     */
    protected function getBulkTransferRequests(): array
    {
        try {
            $bulkTransferRequestsFactory = $this->bulkTransferRequestsManager->createBulkRequest();
            $collection = $bulkTransferRequestsFactory->getCollection();
            
            // Filter out cancelled requests and requests with empty queue_ids
            $collection->addFieldToFilter('is_cancelled', ['neq' => 1])
                      ->addFieldToFilter('queue_ids', ['notnull' => true])
                      ->addFieldToFilter('queue_ids', ['neq' => '']);

            return $collection->getItems();
        } catch (\Exception $e) {
            $this->cronLogger->logRequest('Error getting bulk transfer requests: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Check if there's an active (pending or running) schedule for the given bulk request ID
     *
     * @param int $bulkRequestId
     * @return bool
     */
    protected function hasActiveSchedule(int $bulkRequestId): bool
    {
        try {
            $connection = $this->scheduleResource->getConnection();
            $tableName = $this->scheduleResource->getMainTable();

            // Check for schedules with messages containing this bulk_request_id
            $select = $connection->select()
                ->from($tableName, ['schedule_id'])
                ->where('job_code LIKE ?', 'wtai_grid_transfer%')
                ->where('status IN (?)', [
                    Schedule::STATUS_PENDING,
                    Schedule::STATUS_RUNNING
                ])
                ->where('messages LIKE ?', '%bulk_request_id:' . $bulkRequestId . '%');

            $result = $connection->fetchOne($select);
            
            return !empty($result);
        } catch (\Exception $e) {
            $this->cronLogger->logRequest('Error checking active schedule for bulk request ID ' . $bulkRequestId . ': ' . $e->getMessage());
            return true; // Assume there's an active schedule to avoid creating duplicates
        }
    }

    /**
     * Schedule grid transfer cron job to run on next cron execution
     * This method is based on TransferViaSchedule.php scheduleGridTransferCron method
     *
     * @param int $bulkRequestId
     * @return void
     */
    protected function scheduleGridTransferCron(int $bulkRequestId): void
    {
        try {
            // Find an available job code from wtai_grid_transfer to wtai_grid_transfer_20
            $availableJobCode = $this->findAvailableGridTransferJobCode();
            
            if ($availableJobCode) {
                $schedule = $this->scheduleFactory->create();
                
                // Schedule at the next minute mark using Magento's default pattern
                $currentTime = $this->dateTime->gmtTimestamp();
                $currentSeconds = (int)date('s', $currentTime);
                
                // If we're close to the next minute (46+ seconds), skip to the minute after next
                // This provides buffer time to ensure the cron job has time to start properly
                if ($currentSeconds >= 46) {
                    $nextMinute = $currentTime + 120; // Add 2 minutes (skip next minute, schedule for the one after)
                } else {
                    $nextMinute = $currentTime + 60; // Add 60 seconds to get next minute
                }
                
                $scheduledAt = date('Y-m-d H:i:00', $nextMinute); // Magento's format with :00 seconds
                
                $schedule->setJobCode($availableJobCode)
                    ->setStatus(Schedule::STATUS_PENDING)
                    ->setCreatedAt($this->dateTime->gmtDate())
                    ->setScheduledAt($scheduledAt)
                    ->setMessages('bulk_request_id:' . $bulkRequestId);
                    
                $this->scheduleResource->save($schedule);
                
                $this->cronLogger->logRequest('Scheduled grid transfer for bulk request ID: ' . $bulkRequestId . ' using job code: ' . $availableJobCode . ' at: ' . $scheduledAt);
            } else {
                $this->cronLogger->logRequest('No available job code for bulk request ID: ' . $bulkRequestId . ' - all job codes are busy');
            }
        } catch (\Exception $e) {
            $this->cronLogger->logRequest('Error scheduling grid transfer for bulk request ID ' . $bulkRequestId . ': ' . $e->getMessage());
        }
    }

    /**
     * Find an available grid transfer job code that's not currently running or pending
     * This method is based on TransferViaSchedule.php findAvailableGridTransferJobCode method
     *
     * @return string|null
     */
    protected function findAvailableGridTransferJobCode(): ?string
    {
        try {
            $connection = $this->scheduleResource->getConnection();
            $tableName = $this->scheduleResource->getMainTable();
            
            // Get all currently running or pending job codes for grid transfer jobs
            $select = $connection->select()
                ->from($tableName, ['job_code'])
                ->where('job_code LIKE ?', 'wtai_grid_transfer%')
                ->where('status IN (?)', [
                    Schedule::STATUS_PENDING,
                    Schedule::STATUS_RUNNING
                ]);
                
            $busyJobCodes = $connection->fetchCol($select);
            
            // Check each job code from wtai_grid_transfer to wtai_grid_transfer_20
            $jobCodes = ['wtai_grid_transfer']; // Start with base job code
            for ($i = 1; $i <= 20; $i++) {
                $jobCodes[] = 'wtai_grid_transfer_' . $i;
            }
            
            // Find the first available job code
            foreach ($jobCodes as $jobCode) {
                if (!in_array($jobCode, $busyJobCodes)) {
                    return $jobCode;
                }
            }
            
            return null; // All job codes are busy
            
        } catch (\Exception $e) {
            $this->cronLogger->logRequest('Error finding available job code: ' . $e->getMessage());
            // If there's an error, fallback to the original job code
            return 'wtai_grid_transfer';
        }
    }
}
