/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

define([
    'jquery',
    'WriteTextAI_WriteTextAI/js/model/pro-badge'
], function ($, proBadge) {
    'use strict';

    const freeTrialMembershipType = 'FreeTrial';
    const starterMembershipType = 'Starter';
    const professionalMembershipType = 'Professional';
    const enterpriseMembershipType = 'Enterprise';

    /**
     * Check if any of the provided variables are undefined
     * @param {Array} variables - Array of variables to check
     * @returns {boolean} - True if any variable is undefined
     */
    function hasUndefinedValues(variables)
    {
        for (var i = 0; i < variables.length; i++) {
            if (typeof variables[i] === 'undefined') {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Update pro badge.
     *
     * Removes the importance of code ordering and the need to add "IMPORTANT:" notes.
     * Fixes the issue where banners incorrectly appear when they shouldn't.
     * Resolves the issue where the badge briefly appears and then disappears immediately.
     *
     * @param {Object} data
     * @returns {void}
     */
    function updateProBadge(data)
    {
        var hasProAccess = data.has_pro_access,
            membershipType = data.membership_type,
            unlockedPremium = data.is_unlocked_premium,
            unlockedProfessional = data.is_unlocked_professional,
            unlockedEnterprise = data.is_unlocked_enterprise,
            unlockedFreeTrial = data.is_unlocked_free_trial,
            noCreditsFreeTrial = data.is_no_credits_free_trial;
            
        var conditionConfigs = [
            {
                method: 'showBadge',
                condition: function () {
                    return !hasProAccess && membershipType !== freeTrialMembershipType; },
                requiredVars: [hasProAccess, membershipType, freeTrialMembershipType]
        },
            {
                method: 'showWelcomeStarter',
                condition: function () {
                    return unlockedPremium && !hasProAccess && membershipType === starterMembershipType; },
                requiredVars: [unlockedPremium, hasProAccess, membershipType, starterMembershipType]
        },
            {
                method: 'showWelcomePro',
                condition: function () {
                    return unlockedProfessional && hasProAccess && membershipType === professionalMembershipType; },
                requiredVars: [unlockedProfessional, hasProAccess, membershipType, professionalMembershipType]
        },
            {
                method: 'showWelcomeEnterprise',
                condition: function () {
                    return unlockedEnterprise && hasProAccess && membershipType === enterpriseMembershipType; },
                requiredVars: [unlockedEnterprise, hasProAccess, membershipType, enterpriseMembershipType]
        },
            {
                method: 'showStartTrial',
                condition: function () {
                    return unlockedFreeTrial && hasProAccess && membershipType === freeTrialMembershipType; },
                requiredVars: [unlockedFreeTrial, hasProAccess, membershipType, freeTrialMembershipType]
        },
            {
                method: 'showEndTrial',
                condition: function () {
                    return !hasProAccess && membershipType === freeTrialMembershipType; },
                requiredVars: [hasProAccess, membershipType, freeTrialMembershipType]
        },
            {
                method: 'showNoMoreTrialCredits',
                condition: function () {
                    return noCreditsFreeTrial && membershipType === freeTrialMembershipType; },
                requiredVars: [noCreditsFreeTrial, membershipType, freeTrialMembershipType]
        }
        ];
        
        conditionConfigs.forEach(function (config) {
            if (hasUndefinedValues(config.requiredVars)) {
                return;
            }
            
            if (typeof proBadge[config.method] === 'function') {
                proBadge[config.method](config.condition());
            }
        });
    }

    return {
        updateProBadge: updateProBadge
    };
});
