<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\ViewModel;

use WriteTextAI\WriteTextAI\Model\SettingManager;
use WriteTextAI\WriteTextAI\Helper\Data;
use WriteTextAI\WriteTextAI\ViewModel\Settings;

class TooltipCategory implements \Magento\Framework\View\Element\Block\ArgumentInterface
{
    public const FAQ_URL = 'https://writetext.ai/frequently-asked-questions#credit-cost';

    public const KEYWORDS_GUIDE_URL = 'https://writetext.ai/guide-mastering-keyword-optimization';

    /**
     * @var SettingManager
     */
    protected $settingManager;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var Settings
     */
    protected $settings;

    /**
     * Constructor
     *
     * @param SettingManager $settingManager
     * @param Data $helper
     * @param Settings $settings
     */
    public function __construct(
        SettingManager $settingManager,
        Data $helper,
        Settings $settings
    ) {
        $this->settingManager = $settingManager;
        $this->helper = $helper;
        $this->settings = $settings;
    }

    /**
     * Get tooltip
     *
     * @param string $block
     * @return string
     */
    public function getTooltipCategory($block)
    {
        $tooltip = '';
        switch ($block) {
            case 'wtai.reviewed.category':
                $tooltip = $this->getReviewedTooltipCategory();
                break;
            case 'wtai.keywords.category':
                $tooltip = $this->getKeywordsTooltipCategory();
                break;
            case 'wtai.generate.rewrite.category':
                $tooltip = $this->getGenerateRewriteTooltipCategory();
                break;
            case 'wtai.audience.category':
                $tooltip = $this->getAudienceTooltipCategory();
                break;
            case 'wtai.category.description':
                $tooltip = $this->getCategoryDescriptionTooltip();
                break;
            case 'wtai.category.image':
                $tooltip = $this->getCategoryImageTooltip();
                break;
            case 'wtai.page.title.category':
                $tooltip = $this->getPageTitleTooltipCategory();
                break;
            case 'wtai.page.description.category':
                $tooltip = $this->getPageDescriptionTooltipCategory();
                break;
            case 'wtai.transfer.category':
                $tooltip = $this->getTransferTooltipCategory();
                break;
            case 'wtai.category.intent':
                $tooltip = $this->getCategoryIntentTooltip();
                break;
            case 'wtai.add.keywords.category':
                $tooltip = $this->getAddKeywordsTooltip();
                break;
        }

        return $tooltip;
    }

    /**
     * Get add keywords tooltip
     *
     * @return string
     */
    public function getAddKeywordsTooltip()
    {
        return sprintf('<p>%s</p>', __('(Optional) In the Starter plan, you can manually add'
            . ' target keywords to guide content generation.'))
            . sprintf('<p>%s</p>', __('In the Pro plan, this field is used to enter optional seed keywords that'
                . ' help kickstart AI-powered keyword analysis. These keywords serve as starting points'
                . ' for identifying high-potential terms, but may not always be included in the final pipeline.'
                . ' If you don’t enter any, WriteText.ai will automatically generate seed keywords based'
                . ' on your category name, representative products, online research, and image analysis. Pro users'
                . ' can also create custom pipelines after analysis to directly target specific keywords they want'
                . ' to optimize for.'));
    }

    /**
     * Get representative products tooltip
     *
     * @return string
     */
    public function getRepresentativesTooltip()
    {
        $rules = $this->getRules();
        $maxRepresentativeProducts = $rules['maxRepresentativeProducts'] ?? '';

        $tooltip = '<p>' . sprintf(
            __('Select up to %s products that best represent this category.'
                . ' WriteText.ai will use available data from these products—such as product names,'
                . ' descriptions, and images—as reference when generating the category text.'),
            $maxRepresentativeProducts
        )
            . sprintf("</p><p>%s</p>", __('If enabled, Product Research will also analyze these representative products'
                . ' online to add missing context and improve the quality and relevance of the generated content.'));

        return $tooltip;
    }

    /**
     * Get meta title tooltip for category
     *
     * @return string
     */
    public function getPageTitleTooltipCategory()
    {
        $pageTitleLength = $this->helper->getMaxLengthSettings('page_title');

        return sprintf(__('<p>WriteText.ai aims to generate a title with around'
            . ' %s characters, based on current SEO best practices. The current'
            . ' meta title saved for the category is displayed in the box on'
            . ' the right for your reference only.</p>'), $pageTitleLength)
            . sprintf("<p>%s</p>", __('To customize how titles are structured or written,'
            . ' you can set up a template for meta titles in the backend under Templates.'
            . ' If you have not set up or assigned any templates, WriteText.ai will'
            . ' generate a title based on its default approach using your category data'));
    }

    /**
     * Get meta description tooltip for category
     *
     * @return string
     */
    public function getPageDescriptionTooltipCategory()
    {
        $maxPageDescriptionLength = $this->helper->getMaxLengthSettings('page_description');

        return sprintf(__('<p>WriteText.ai aims to generate a description with'
            . ' around %s characters, based on current SEO best practices. The current'
            . ' meta description saved for the category is displayed in the box on'
            . ' the right for your reference only.</p>'), $maxPageDescriptionLength);
    }

    /**
     * Get generate rewrite tooltip category
     *
     * @return string
     */
    public function getGenerateRewriteTooltipCategory()
    {
        return sprintf('<p class="wtai-heading">%s</p>', __('Generate new text'))
            . sprintf('<p>%s</p>', __('Generate text for your selected text types based on'
                . ' the settings on this page.'));
    }

    /**
     * Get transfer tooltip for Category
     *
     * @return string
     */
    public function getTransferTooltipCategory()
    {
        return sprintf("<p>%s</p>", __("When you're done generating and/or editing text,"
                . " you have the option to save the draft inside WriteText.ai or transfer"
                . " the text to Magento. Transferring your text to Magento will publish it"
                . " on the website."))
            . sprintf("<p>%s</p>", __("Note: Any media or directive you have inserted in"
                . " your current Magento text will be overwritten when you transfer from WriteText.ai."));
    }

    /**
     * Get reviewed tooltip for category
     *
     * @return string
     */
    public function getReviewedTooltipCategory()
    {
        return sprintf('<p>%s</p>', __('Check this box to keep track of the categories'
                . ' where you have reviewed the text. This is especially helpful if you'
                . ' have an internal workflow where text needs to go through a review'
                . ' process first before being published on the website. This checkbox'
                . ' does not affect the live content, it is only a classification.'))
            . sprintf('<p>%s</p>', __('You can filter reviewed categories by'
                . ' selecting “Reviewed” under the “Filter by WriteText.ai status”'
                . ' dropdown in the category list.'));
    }

    /**
     * Get audience tooltip for category
     *
     * @return string
     */
    public function getAudienceTooltipCategory()
    {
        return __('<p>Select the applicable audience for your category so WriteText.ai'
            . ' can generate text that will appeal to them.  If you don’t select an audience,'
            . ' the generated text will default to a “neutral” audience. WriteText.ai also'
            . ' suggests more specific target markets based on your keywords. Click on the'
            . ' audience to select it or type in your own custom audience in the box.</p>');
    }

    /**
     * Get keywords tooltip
     *
     * @return string
     */
    public function getKeywordsTooltipCategory()
    {
        // phpcs:disable Generic.Files.LineLength
        return sprintf("<p><strong>%s</strong></p>", __('Target keywords'))
            . sprintf("<p>%s</p>", __('Target keywords allow you to guide the AI toward optimizing your content for specific search terms. How you can use them depends on your plan.'))
            . sprintf("<p><strong>%s</strong></p>", __('Starter Plan'))
            . sprintf("<p>%s</p>", __('In the Starter plan, you can manually add target keywords before generating content. This is ideal if you already know which keywords you want to focus on. However, the full keyword optimization pipeline—including keyword analysis, ranking insights, clustering, and automation—is only available in the Pro plan.'))
            . sprintf("<p><strong>%s</strong></p>", __('Pro Plan'))
            . sprintf("<p>%s", __('Pro unlocks the full keyword optimization pipeline'))
            . sprintf(" (<a href='%s' target='_blank'>%s</a>): ", self::KEYWORDS_GUIDE_URL, __('read the complete guide here'))
            . sprintf("%s</p>", __('an AI-powered keyword analysis tool that uses the keyword ladder strategy to help you gradually improve your category page rankings.'))
            . sprintf("<p>%s</p>", __('Before starting, you can optionally add seed keywords—these help guide the AI during analysis, but aren’t guaranteed to be used if they don’t match the category context.'))
            . sprintf("<p>%s</p>", __('Once you click "Start AI-powered keyword analysis", the system will:'))
            . sprintf("<p><ul><li class='wtai-list-disc'>%s</li>", __('Discover keywords your category pages already rank for'))
            . sprintf("<li class='wtai-list-disc'>%s</li>", __('Suggest new keyword ideas using category data, image analysis, and online research'))
            . sprintf("<li class='wtai-list-disc'>%s</li>", __('Automatically organize keyword suggestions into cluster-based pipelines, starting with low-difficulty terms and progressing to higher-volume, more competitive ones'))
            . sprintf("<li class='wtai-list-disc'>%s</li>", __('Allow you to create your own custom pipelines to set specific keywords and stages for targeting'))
            . sprintf("<li class='wtai-list-disc'>%s</li></ul></p>", __('Provide keyword insights such as search volume, competition level, and search intent'))
            . sprintf("<p>%s</p>", __('You can continue to manually add target keywords as needed to further refine the AI’s output.'))
            . sprintf("<p>%s</p>", __('Due to the complex nature of this process, starting AI-powered keyword analysis may take some time. You can click on "Automatically generate text when keyword analysis is done" to streamline the process. Make sure to select the text types you want to generate before starting AI-powered keyword analysis.'))
            . sprintf("<p><b>%s</b></p>", __('How the keyword optimization pipeline works'))
            . sprintf("<p>%s</p>", __('<em>Starting with low-difficulty keywords:</em><br>The pipeline identifies keywords with lower competition but good search potential.'))
            . sprintf("<p>%s</p>", __('<em>Progressive optimization:</em><br>As your category page starts to rank for low-difficulty keywords, the pipeline gradually recommends optimizing for higher-difficulty keywords.'))
            . sprintf("<p>%s</p>", __('<em>Automation and notifications:</em><br>WriteText.ai automates much of this process, analyzing keyword trends and notifying you when pages are ready for the next optimization step. Depending on your preferences, it can fully automate regenerating text with updated keywords and transferring to Magento, just regenerate and notify you for review, or prompt you to check and initiate regenerating text yourself.'))
            . sprintf("<p>%s</p>", __('<em>Iterative content refinement:</em><br>At every stage, the pipeline ensures your content is updated and optimized for current keyword trends, helping you maintain relevance and visibility.'));
        // phpcs:enable Generic.Files.LineLength
    }

    /**
     * Get category description tooltip
     *
     * @return string
     */
    public function getCategoryDescriptionTooltip()
    {
        return __('<p>The current category description saved for the category is displayed'
                . ' in the box on the right for your reference only.</p>')
            . __('<p>Target length<br/>')
            . __('Indicate your target length by setting a minimum and maximum word'
                . ' count. WriteText.ai will aim to generate text within the number'
                . ' you set, but it may give you more or less words than expected from'
                . ' time to time. Note that your credit cost will depend on the target'
                . ' length you set so make sure to set a reasonable target range. This'
                . " target length is only considered when you don't have a template selected.</p>")
            . __('<p>Templates<br/>')
            . __('To better customize how descriptions are structured or written, you'
                . ' can set up a template for category descriptions in the backend'
                . ' under Templates. If you haven’t set or assigned a template, WriteText.ai'
                . ' will generate the text based on its default approach using your category'
                . ' data and the target length you define. When a template is used, the'
                . ' target length setting is ignored and content is generated based on'
                . ' the structure of the template. An estimated credit cost is shown in'
                . ' the preview panel based on the content sections included.');
    }

    /**
     * Get category image tooltip
     *
     * @return string
     */
    public function getCategoryImageTooltip()
    {
        return __('<p>If available, you can let AI analyze the category image in order'
                .' to generate more accurate and relevant text. Please check that the image'
                .' accurately represents the category (i.e., it is not some kind of'
                .' placeholder or a generic image for your shop).</p>');
    }

    /**
     * Get intent tooltip
     *
     * @return string
     */
    public function getCategoryIntentTooltip()
    {
        return __("Search intent refers to the purpose behind a user's search"
            . " query—what they’re looking to achieve. By default, commercial"
            . " intent is selected as recommended for category pages.");
    }
    /**
     * Get intent tooltip
     *
     * @return string
     */
    public function getCategoryIntentTooltipDisabled()
    {
        return __("Search intent refers to the purpose behind a user's search"
            . " query—what they’re looking to achieve. By default, commercial"
            . " intent is selected as recommended for category pages.");
    }
}
