<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Model;

use WriteTextAI\WriteTextAI\Model\SingleGenerateRequestsFactory;
use WriteTextAI\WriteTextAI\Api\SingleGenerateRequestsRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ApiManager;
use WriteTextAI\WriteTextAI\Model\Api\Keywords;

class SingleGenerateRequestsManager
{
    /**
     * @var SingleGenerateRequestsFactory
     */
    protected $singleGenerateRequestsFactory;

    /**
     * @var SingleGenerateRequestsRepositoryInterface
     */
    protected $singleGenerateRequestsRepository;

    /**
     * @var ApiManager
     */
    protected $apiManager;

    /**
     * @var Keywords
     */
    protected $keywords;

    /**
     * Constructor
     *
     * @param SingleGenerateRequestsFactory $singleGenerateRequestsFactory
     * @param SingleGenerateRequestsRepositoryInterface $singleGenerateRequestsRepository
     * @param ApiManager $apiManager
     * @param Keywords $keywords
     */
    public function __construct(
        SingleGenerateRequestsFactory $singleGenerateRequestsFactory,
        SingleGenerateRequestsRepositoryInterface $singleGenerateRequestsRepository,
        ApiManager $apiManager,
        Keywords $keywords
    ) {
        $this->singleGenerateRequestsFactory = $singleGenerateRequestsFactory;
        $this->singleGenerateRequestsRepository = $singleGenerateRequestsRepository;
        $this->apiManager = $apiManager;
        $this->keywords = $keywords;
    }

    /**
     * Set single generate request
     *
     * @param string $requestId
     * @param string $user
     * @param string $entityType
     * @param string $storeId
     * @param string $recordId = null
     * @return void
     */
    public function setSingleGenerateRequest($requestId, $user, $entityType, $storeId, $recordId = null)
    {
        $singleGenerateRequests = $this->singleGenerateRequestsFactory->create();

        $existing = $singleGenerateRequests->getCollection()
            ->addFieldToFilter('request_id', $requestId)
            ->getFirstItem();

        if ($existing->getId()) {
            // Update the existing record
            $existing->setRequestId($requestId);
            $existing->setEntityType($entityType);
            $existing->setStoreId($storeId);
            $existing->setRecordId($recordId);
            $singleGenerateRequests = $existing;
        } else {
            // Record not found, create a new one
            $singleGenerateRequests->setData([
                "user" => $user,
                "request_id" => $requestId,
                "entity_type" => $entityType,
                "store_id" => $storeId,
                "record_id" => $recordId
            ]);
        }

        $this->singleGenerateRequestsRepository->save($singleGenerateRequests);
    }

    /**
     * Delete single generate request
     *
     * @param string $requestId
     * @return void
     */
    public function deleteSingleGenerateRequest($requestId)
    {
        $singleGenerateRequests = $this->singleGenerateRequestsFactory->create();

        $existing = $singleGenerateRequests->getCollection()
            ->addFieldToFilter('request_id', $requestId)
            ->getFirstItem();

        if ($existing->getId()) {
            $this->singleGenerateRequestsRepository->delete($existing);
        }
    }

    /**
     * Delete single generate request by user
     *
     * @param string $user
     * @return void
     */
    public function deleteSingleGenerateRequestByUser($user)
    {
        $singleGenerateRequests = $this->singleGenerateRequestsFactory->create();

        $existing = $singleGenerateRequests->getCollection()
            ->addFieldToFilter('user', $user)
            ->getFirstItem();

        if ($existing->getId()) {
            $this->singleGenerateRequestsRepository->delete($existing);
        }
    }

    /**
     * Delete all single generate requests
     *
     * @return void
     */
    public function deleteAllSingleGenerateRequests()
    {
        $singleGenerateRequests = $this->singleGenerateRequestsFactory->create();
        $singleGenerateRequests->getCollection()->walk('delete');
    }

    /**
     * Get single generate requests
     *
     * @param string $requestId
     *
     * @return array
     */
    public function getSingleGenerateRequestsByRequestId($requestId)
    {
        $singleGenerateRequests = $this->singleGenerateRequestsFactory->create();
        $singleGenerateRequests = $singleGenerateRequests->getCollection()
            ->addFieldToFilter('request_id', $requestId)
            ->getFirstItem();

        if ($singleGenerateRequests->getId()) {
            return $singleGenerateRequests->getData();
        }

        return [];
    }
}
