<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Model;

use WriteTextAI\WriteTextAI\Model\SettingFactory;
use WriteTextAI\WriteTextAI\Api\SettingRepositoryInterface;
use WriteTextAI\WriteTextAI\Model\ApiManager;
use WriteTextAI\WriteTextAI\Model\Api\Session as ApiSession;

class SettingManager
{
    public const IMPLEMENTED_ETAGS = [
        'rules',
        'tones',
        'styles',
        'audiences',
        'credits',
        'formalLanguageSupport',
        'formalLanguages',
        'disallowedCombinations'
    ];

    public const LOCALIZED_SETTINGS = [
        'tones',
        'styles',
        'audiences'
    ];

    /**
     * @var SettingFactory
     */
    protected $settingFactory;

    /**
     * @var SettingRepositoryInterface
     */
    protected $settingRepository;

    /**
     * @var ApiManager
     */
    protected $apiManager;

    /**
     * @var ApiSession
     */
    protected $apiSession;

    /**
     * Constructor
     *
     * @param SettingFactory $settingFactory
     * @param SettingRepositoryInterface $settingRepository
     * @param ApiManager $apiManager
     * @param ApiSession $apiSession
     */
    public function __construct(
        SettingFactory $settingFactory,
        SettingRepositoryInterface $settingRepository,
        ApiManager $apiManager,
        ApiSession $apiSession
    ) {
        $this->settingFactory = $settingFactory;
        $this->settingRepository = $settingRepository;
        $this->apiManager = $apiManager;
        $this->apiSession = $apiSession;
    }

    /**
     * Sync settings with API
     *
     * @return void
     */
    public function syncSettings()
    {
        $settings = $this->settingFactory->create();
        $etags = $this->apiManager->getEtags();

        if (empty($etags)) {
            return;
        }

        // Query once for all settings
        $existingSettings = $settings->getCollection()
            ->addFieldToFilter('setting_name', ['in' => array_keys($etags)])
            ->getItems();

        // Convert to associative array for easy lookup
        $existingSettingsAssoc = [];
        foreach ($existingSettings as $existingSetting) {
            $existingSettingsAssoc[$existingSetting->getSettingName()] = $existingSetting;
        }
        
        foreach ($etags as $name => $etag) {
            if (!in_array($name, self::IMPLEMENTED_ETAGS)) {
                continue;
            }

            if ($name == 'formalLanguages') {
                $locales = $this->getSettings('formalLanguageSupport');
                foreach ($locales as $locale) {
                    $this->syncSetting($name, $locale, $etag);
                }
                continue;
            }

            if (in_array($name, self::LOCALIZED_SETTINGS)) {
                $locale = $this->apiSession->getRegionIndependentLanguage();
                $this->syncSetting($name, $locale, $etag);
                continue;
            }

            if (isset($existingSettingsAssoc[$name])) {
                // Update the existing record
                $existing = $existingSettingsAssoc[$name];
                if ($existing->getEtag() != $etag) {
                    $existing->setValue($this->getNewValue($name));
                    $existing->setEtag($etag);
                    $settings = $existing;
                    $this->settingRepository->save($settings);
                }
            } else {
                // Record not found, create a new one
                $value = $this->getNewValue($name);
                if ($value != null) {
                    $settings->setData([
                        "setting_name" => $name,
                        "value" => $value,
                        "etag" => $etag
                    ]);
                    $this->settingRepository->save($settings);
                }
            }
        }
    }
    
    /**
     * Get settings value
     *
     * @param string $name
     * @param string $locale
     * @return array
     */
    public function getSettings($name, $locale = null)
    {
        $settings = $this->settingFactory->create();
        $settings = $settings->getCollection()
            ->addFieldToFilter('setting_name', $name);

        if ($locale) {
            $settings->addFieldToFilter('locale', $locale);
        }

        $settings = $settings->getFirstItem();

        if ($settings->getSettingId()) {
            return json_decode($settings->getValue(), true);
        } else {
            $value = $this->getNewValue($name, $locale);

            if ($value != null && $name != 'formalLanguages') {
                $settings = $this->settingFactory->create();
                $settings->setData([
                    "setting_name" => $name,
                    "value" => $value
                ]);
                $this->settingRepository->save($settings);
            }

            return $value;
        }
    }
    
    /**
     * Get new value from API
     *
     * @param string $name
     * @param string $locale
     * @return string||null
     */
    private function getNewValue($name, $locale = null)
    {
        $value = null;
        switch ($name) {
            case 'rules':
                $value = $this->apiManager->getRules();
                break;
            case 'tones':
                $value = $this->apiManager->getTones($locale);
                break;
            case 'styles':
                $value = $this->apiManager->getStyles($locale);
                break;
            case 'audiences':
                $value = $this->apiManager->getAudiences($locale);
                break;
            case 'credits':
                $value = $this->apiManager->getCredits();
                break;
            case 'formalLanguageSupport':
                $value = $this->apiManager->getFormalLanguageSupport();
                break;
            case 'formalLanguages':
                $value = $this->apiManager->getFormalLanguages($locale);
                break;
            case 'disallowedCombinations':
                $value = $this->apiManager->getDisallowedCombinations();
                break;
            case 'locations':
                $value = $this->apiManager->getLocations();
                break;
        }

        if (is_array($value)) {
            $value = json_encode($value);
        }

        return $value;
    }
    
    /**
     * Save settings
     *
     * @param string $name
     * @param string $locale
     * @param string $etag
     * @return void
     */
    public function syncSetting(
        $name,
        $locale = null,
        $etag = null
    ) {
        $settings = $this->settingFactory->create();

        $existing = $settings->getCollection()
            ->addFieldToFilter('setting_name', $name);

        if ($locale) {
            $existing->addFieldToFilter('locale', $locale);
        }

        $existing = $existing->getFirstItem();

        if ($existing && $existing->getSettingId()) {
            if ($etag) {
                if ($existing->getEtag() != $etag) {
                    $existing->setValue($this->getNewValue($name, $locale));
                    $existing->setEtag($etag);
                    $settings = $existing;
                    $this->settingRepository->save($settings);
                }
            } else {
                $existing->setValue($this->getNewValue($name, $locale));
                $settings = $existing;
                $this->settingRepository->save($settings);
            }
        } else {
            $value = $this->getNewValue($name, $locale);
            if ($value != null) {
                $data = [
                    "setting_name" => $name,
                    "value" => $value
                ];

                if ($locale) {
                    $data['locale'] = $locale;
                }

                if ($etag) {
                    $data['etag'] = $etag;
                }

                $settings->setData($data);

                $this->settingRepository->save($settings);
            }
        }
    }
}
