<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

declare(strict_types=1);

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\BulkTransferRequestsRepositoryInterface;
use WriteTextAI\WriteTextAI\Api\Data\BulkTransferRequestsInterface;
use WriteTextAI\WriteTextAI\Api\Data\BulkTransferRequestsInterfaceFactory;
use WriteTextAI\WriteTextAI\Api\Data\BulkTransferRequestsSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Model\ResourceModel\BulkTransferRequests as ResourceBulkTransferRequests;
use WriteTextAI\WriteTextAI\Model\ResourceModel\BulkTransferRequests\CollectionFactory
    as BulkTransferRequestsCollectionFactory;

class BulkTransferRequestsRepository implements BulkTransferRequestsRepositoryInterface
{
    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var BulkTransferRequestsCollectionFactory
     */
    protected $bulkTransferRequestsCollectionFactory;

    /**
     * @var BulkTransferRequestsInterfaceFactory
     */
    protected $bulkTransferRequestsFactory;

    /**
     * @var BulkTransferRequests
     */
    protected $searchResultsFactory;

    /**
     * @var ResourceBulkTransferRequests
     */
    protected $resource;

    /**
     * @param ResourceBulkTransferRequests $resource
     * @param BulkTransferRequestsInterfaceFactory $bulkTransferRequestsFactory
     * @param BulkTransferRequestsCollectionFactory $bulkTransferRequestsCollectionFactory
     * @param BulkTransferRequestsSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceBulkTransferRequests $resource,
        BulkTransferRequestsInterfaceFactory $bulkTransferRequestsFactory,
        BulkTransferRequestsCollectionFactory $bulkTransferRequestsCollectionFactory,
        BulkTransferRequestsSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->bulkTransferRequestsFactory = $bulkTransferRequestsFactory;
        $this->bulkTransferRequestsCollectionFactory = $bulkTransferRequestsCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(
        BulkTransferRequestsInterface $bulkTransferRequests
    ) {
        try {
            $this->resource->save($bulkTransferRequests);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the bulkTransferRequests: %1',
                $exception->getMessage()
            ));
        }
        return $bulkTransferRequests;
    }

    /**
     * @inheritDoc
     */
    public function get($bulkTransferRequestsId)
    {
        $bulkTransferRequests = $this->bulkTransferRequestsFactory->create();
        $this->resource->load($bulkTransferRequests, $bulkTransferRequestsId);
        if (!$bulkTransferRequests->getId()) {
            throw new NoSuchEntityException(__(
                'bulk_transfer_requests with id "%1" does not exist.',
                $bulkTransferRequestsId
            ));
        }
        return $bulkTransferRequests;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->bulkTransferRequestsCollectionFactory->create();
        
        $this->collectionProcessor->process($criteria, $collection);
        
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        
        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }
        
        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(
        BulkTransferRequestsInterface $bulkTransferRequests
    ) {
        try {
            $bulkTransferRequestsModel = $this->bulkTransferRequestsFactory->create();
            $this->resource->load($bulkTransferRequestsModel, $bulkTransferRequests->getBulkTransferRequestsId());
            $this->resource->delete($bulkTransferRequestsModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the bulk_transfer_requests: %1',
                $exception->getMessage()
            ));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($bulkTransferRequestsId)
    {
        return $this->delete($this->get($bulkTransferRequestsId));
    }
}
