<?php
/**
 * @author 1902 Software
 * @copyright Copyright © 2023 1902 Software (https://1902software.com/magento/)
 * @package WriteTextAI_WriteTextAI
 */

namespace WriteTextAI\WriteTextAI\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use WriteTextAI\WriteTextAI\Api\AiProductRepositoryInterface;
use WriteTextAI\WriteTextAI\Api\Data\AiProductSearchResultsInterfaceFactory;
use WriteTextAI\WriteTextAI\Model\ResourceModel\AiProduct as ResourceAiProduct;
use WriteTextAI\WriteTextAI\Model\ResourceModel\AiProduct\CollectionFactory as AiProductCollectionFactory;

class AiProductRepository implements AiProductRepositoryInterface
{
    /**
     * @var ResourceAiProduct
     */
    protected $resource;

    /**
     * @var AiProductFactory
     */
    protected $aiProductFactory;

    /**
     * @var AiProductCollectionFactory
     */
    protected $aiProductCollectionFactory;

    /**
     * @var AiProductSearchResultsInterfaceFactory
     */
    protected $searchResultsFactory;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @param ResourceAiProduct $resource
     * @param AiProductFactory $aiProductFactory
     * @param AiProductCollectionFactory $aiProductCollectionFactory
     * @param AiProductSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceAiProduct $resource,
        AiProductFactory $aiProductFactory,
        AiProductCollectionFactory $aiProductCollectionFactory,
        AiProductSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->aiProductFactory = $aiProductFactory;
        $this->aiProductCollectionFactory = $aiProductCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(
        \WriteTextAI\WriteTextAI\Api\Data\AiProductInterface $product
    ) {
        try {
            $this->resource->save($product);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $product;
    }

    /**
     * @inheritDoc
     */
    public function get($productsId)
    {
        $products = $this->aiProductFactory->create();
        $this->resource->load($products, $productsId, 'product_id');
        if (!$products->getId()) {
            throw new NoSuchEntityException(__('The product with the "%1" ID doesn\'t exist.', $productsId));
        }
        return $products;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $searchCriteria
    ) {
        /** @var \WriteTextAI\WriteTextAI\Model\ResourceModel\AiProduct\Collection $collection */
        $collection = $this->aiProductCollectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        /** @var AiProductSearchResultsInterface $searchResults */
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(
        \WriteTextAI\WriteTextAI\Api\Data\AiProductInterface $products
    ) {
        try {
            $this->resource->delete($products);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($productsId)
    {
        return $this->delete($this->getById($productsId));
    }
}
